/* app.js — CRM global scripts */
'use strict';

// ── Theme toggle ──────────────────────────────────────────
const themeBtn = document.getElementById('themeToggle');
if (themeBtn) {
  themeBtn.addEventListener('click', () => {
    const html  = document.documentElement;
    const isDark = html.getAttribute('data-bs-theme') === 'dark';
    const next  = isDark ? 'light' : 'dark';
    html.setAttribute('data-bs-theme', next);
    document.cookie = 'crm_theme=' + next + ';path=/;max-age=31536000';
    themeBtn.querySelector('i').className = 'bi bi-' + (next === 'dark' ? 'sun' : 'moon');
  });
}

// ── Sidebar collapse (desktop) ────────────────────────────
const sidebar       = document.getElementById('sidebar');
const sidebarToggle = document.getElementById('sidebarToggle');
const mobileTgl     = document.getElementById('sidebarToggleMobile');

if (sidebarToggle && sidebar) {
  const stored = localStorage.getItem('sidebarCollapsed');
  if (stored === '1') sidebar.classList.add('collapsed');
  sidebarToggle.addEventListener('click', () => {
    sidebar.classList.toggle('collapsed');
    localStorage.setItem('sidebarCollapsed', sidebar.classList.contains('collapsed') ? '1' : '0');
  });
}

if (mobileTgl && sidebar) {
  mobileTgl.addEventListener('click', () => sidebar.classList.toggle('mobile-open'));
  document.addEventListener('click', e => {
    if (sidebar.classList.contains('mobile-open') && !sidebar.contains(e.target) && e.target !== mobileTgl)
      sidebar.classList.remove('mobile-open');
  });
}

// ── Phone auto-format ─────────────────────────────────────
document.querySelectorAll('input[data-fmt="phone"]').forEach(el => {
  el.addEventListener('input', () => {
    let v = el.value.replace(/\D/g,'').slice(0,10);
    if (v.length >= 7)      v = '('+v.slice(0,3)+') '+v.slice(3,6)+'-'+v.slice(6);
    else if (v.length >= 4) v = '('+v.slice(0,3)+') '+v.slice(3);
    else if (v.length > 0)  v = '('+v;
    el.value = v;
  });
});

// ── Name auto-capitalize ──────────────────────────────────
document.querySelectorAll('input[data-fmt="capitalize"]').forEach(el => {
  el.addEventListener('blur', () => {
    el.value = el.value.replace(/\b\w/g, c => c.toUpperCase());
  });
});

// ── BIN lookup ────────────────────────────────────────────
const binInput  = document.getElementById('card_number_bin');
const binStatus = document.getElementById('bin_status');

if (binInput && binStatus) {
  let binTimer;
  binInput.addEventListener('input', () => {
    clearTimeout(binTimer);
    const val = binInput.value.replace(/\D/g,'');
    if (val.length < 6) { binStatus.innerHTML = ''; return; }
    binStatus.innerHTML = '<span class="bin-status"><i class="bi bi-hourglass-split"></i> Looking up…</span>';
    binTimer = setTimeout(() => {
      fetch('../../api/bin_lookup.php?bin='+encodeURIComponent(val.slice(0,6)))
        .then(r => r.json())
        .then(d => {
          if (d.brand) {
            binStatus.innerHTML = '<span class="bin-status"><i class="bi bi-check-circle-fill text-success"></i> '+
              escHtml(d.brand.toUpperCase())+' '+escHtml(d.type||'')+
              (d.issuer?' — '+escHtml(d.issuer):'')+' </span>';
            if (document.getElementById('card_brand'))  document.getElementById('card_brand').value  = d.brand;
            if (document.getElementById('card_issuer')) document.getElementById('card_issuer').value = d.issuer||'';
            if (document.getElementById('card_type_detail')) document.getElementById('card_type_detail').value = d.type||'';
          } else {
            binStatus.innerHTML = '<span class="bin-status text-muted"><i class="bi bi-question-circle"></i> Not found</span>';
          }
        }).catch(() => { binStatus.innerHTML = ''; });
    }, 500);
  });
}

// ── ZIP → City/State autocomplete ─────────────────────────
const zipInput = document.querySelector('input[name="zip"]');
if (zipInput) {
  zipInput.addEventListener('blur', () => {
    const z = zipInput.value.replace(/\D/g,'');
    if (z.length !== 5) return;
    fetch('https://api.zippopotam.us/us/'+z)
      .then(r => r.json())
      .then(d => {
        if (!d.places || !d.places[0]) return;
        const city  = document.querySelector('input[name="city"]');
        const state = document.querySelector('select[name="state"]');
        if (city  && !city.value)  city.value  = d.places[0]['place name'];
        if (state && !state.value) state.value = d.places[0]['state abbreviation'];
      }).catch(()=>{});
  });
}

// ── Confirm dialogs ───────────────────────────────────────
document.querySelectorAll('[data-confirm]').forEach(el => {
  el.addEventListener('click', e => {
    if (!confirm(el.dataset.confirm)) e.preventDefault();
  });
});

// ── Select-all checkbox ───────────────────────────────────
const selectAll = document.getElementById('selectAll');
if (selectAll) {
  selectAll.addEventListener('change', () => {
    document.querySelectorAll('.row-check').forEach(cb => cb.checked = selectAll.checked);
  });
}

// ── Utility ───────────────────────────────────────────────
function escHtml(str) {
  return String(str).replace(/&/g,'&amp;').replace(/</g,'&lt;').replace(/>/g,'&gt;').replace(/"/g,'&quot;');
}

// ── Sort table columns ────────────────────────────────────
document.querySelectorAll('th[data-sort]').forEach(th => {
  th.style.cursor = 'pointer';
  th.addEventListener('click', () => {
    const url = new URL(window.location.href);
    const cur = url.searchParams.get('sort');
    const curDir = url.searchParams.get('dir') || 'asc';
    const col = th.dataset.sort;
    url.searchParams.set('sort', col);
    url.searchParams.set('dir', (cur===col && curDir==='asc') ? 'desc' : 'asc');
    url.searchParams.set('page','1');
    window.location.href = url.toString();
  });
});
