<?php
function h(mixed $v): string {
    return htmlspecialchars((string)($v ?? ''), ENT_QUOTES|ENT_SUBSTITUTE, 'UTF-8');
}
function redirect(string $url): never { header('Location: '.$url); exit; }

function generate_ref(): string {
    return 'CC-'.strtoupper(substr(md5(uniqid('',true)),0,8));
}
function format_phone(string $p): string {
    $d = preg_replace('/\D/','',$p);
    return strlen($d)===10 ? '('.substr($d,0,3).') '.substr($d,3,3).'-'.substr($d,6) : $p;
}
function format_currency(?float $v): string {
    return $v===null ? '—' : '$'.number_format($v,2);
}
function status_badge(string $s): string {
    $map = ['new'=>'badge-new','in_review'=>'badge-review','approved'=>'badge-approved',
            'declined'=>'badge-declined','forwarded'=>'badge-forwarded','on_hold'=>'badge-hold'];
    $cls = $map[$s] ?? 'bg-secondary';
    return '<span class="status-badge '.$cls.'">'.h(ucfirst(str_replace('_',' ',$s))).'</span>';
}
function flash(string $k, string $m): void { $_SESSION['flash'][$k]=$m; }
function get_flash(string $k): ?string {
    $m=$_SESSION['flash'][$k]??null; unset($_SESSION['flash'][$k]); return $m;
}
function csrf_token(): string {
    if (empty($_SESSION['csrf_token'])) $_SESSION['csrf_token']=bin2hex(random_bytes(32));
    return $_SESSION['csrf_token'];
}
function csrf_check(): void {
    if (!hash_equals($_SESSION['csrf_token']??'', $_POST['csrf_token']??'')) {
        http_response_code(403); die('CSRF validation failed.');
    }
}

/** Get custom fields, optionally only active ones. Cached per request. */
function get_custom_fields(bool $activeOnly=true): array {
    static $cache = [];
    $key = $activeOnly ? 'active' : 'all';
    if (!isset($cache[$key])) {
        $sql = "SELECT * FROM custom_fields".($activeOnly?" WHERE is_active=1":"")." ORDER BY sort_order,id";
        $cache[$key] = db()->query($sql)->fetchAll();
    }
    return $cache[$key];
}

/** Invalidate custom fields cache (call after insert/update/delete) */
function bust_cf_cache(): void {
    // PHP static cache is per-request anyway; this is a hook for future Redis/APCu
}

function get_lead_custom_values(int $leadId): array {
    $st = db()->prepare("SELECT cf.field_key, lcv.value FROM lead_custom_values lcv
        JOIN custom_fields cf ON cf.id=lcv.custom_field_id WHERE lcv.lead_id=?");
    $st->execute([$leadId]);
    $out=[];
    foreach($st->fetchAll() as $r) $out[$r['field_key']]=$r['value'];
    return $out;
}

function save_custom_values(int $leadId, array $fields, array $post): void {
    foreach($fields as $f) {
        $val = $post['cf_'.$f['field_key']] ?? null;
        if(is_array($val)) $val=implode(',',$val);
        $st = db()->prepare("INSERT INTO lead_custom_values (lead_id,custom_field_id,value) VALUES (?,?,?)
            ON DUPLICATE KEY UPDATE value=VALUES(value)");
        $st->execute([$leadId,$f['id'],$val]);
    }
}

/** Can this role manage custom fields? */
function can_manage_fields(): bool {
    $role = current_user()['role'] ?? '';
    return in_array($role, ['agent','processor','admin']);
}

/** Can this role access bulk upload? Admin + processor */
function can_bulk_upload(): bool {
    return in_array(current_user()['role'] ?? '', ['admin','processor']);
}

/** US states */
function us_states(): array {
    return ['AL','AK','AZ','AR','CA','CO','CT','DE','FL','GA','HI','ID','IL','IN','IA',
            'KS','KY','LA','ME','MD','MA','MI','MN','MS','MO','MT','NE','NV','NH','NJ',
            'NM','NY','NC','ND','OH','OK','OR','PA','RI','SC','SD','TN','TX','UT','VT',
            'VA','WA','WV','WI','WY','DC'];
}

/** Build a safe ORDER BY column for leads queries */
function safe_sort_col(string $col): string {
    $allowed = ['created_at','updated_at','last_name','first_name','status','card_issuer','email','phone'];
    return in_array($col,$allowed) ? $col : 'created_at';
}

/** Stream CSV export — call before any output */
function export_leads_csv(array $leads, array $customFields): never {
    header('Content-Type: text/csv');
    header('Content-Disposition: attachment; filename="leads_'.date('Ymd_His').'.csv"');
    $out = fopen('php://output','w');
    // Header row
    $headers = ['Ref','First Name','Last Name','Email','Phone','Address','City','State','ZIP',
                'DOB','SSN Last4','Annual Income','Employment','Credit Score','Requested Limit',
                'Card Type','BIN','Brand','Issuer','Card Type Detail','Status','Notes','Agent','Created'];
    foreach($customFields as $cf) $headers[]=$cf['label'];
    fputcsv($out,$headers);
    foreach($leads as $l) {
        $row = [
            $l['ref_number'],$l['first_name'],$l['last_name'],$l['email'],$l['phone'],
            $l['address'],$l['city'],$l['state'],$l['zip'],$l['dob'],$l['ssn_last4'],
            $l['annual_income'],$l['employment_status'],$l['credit_score_range'],$l['requested_limit'],
            $l['card_type'],$l['card_number_bin'],$l['card_brand'],$l['card_issuer'],$l['card_type_detail'],
            $l['status'],$l['notes'],$l['agent_name']??$l['agent_username']??'',$l['created_at'],
        ];
        foreach($customFields as $cf) $row[] = $l['cv_'.$cf['field_key']] ?? '';
        fputcsv($out,$row);
    }
    fclose($out);
    exit;
}

