-- ============================================================
--  Financial Leads CRM — Database Schema
-- ============================================================
SET SQL_MODE = "NO_AUTO_VALUE_ON_ZERO";
SET time_zone = "+00:00";

CREATE TABLE IF NOT EXISTS `users` (
  `id`            INT UNSIGNED     NOT NULL AUTO_INCREMENT,
  `username`      VARCHAR(80)      NOT NULL,
  `email`         VARCHAR(180)     NOT NULL,
  `password_hash` VARCHAR(255)     NOT NULL,
  `role`          ENUM('agent','processor','admin') NOT NULL DEFAULT 'agent',
  `full_name`     VARCHAR(160)     NOT NULL DEFAULT '',
  `is_active`     TINYINT(1)       NOT NULL DEFAULT 1,
  `last_login`    DATETIME                  DEFAULT NULL,
  `created_at`    DATETIME         NOT NULL DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `uq_username` (`username`),
  UNIQUE KEY `uq_email`    (`email`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

CREATE TABLE IF NOT EXISTS `leads` (
  `id`                 INT UNSIGNED  NOT NULL AUTO_INCREMENT,
  `ref_number`         VARCHAR(20)   NOT NULL,
  `first_name`         VARCHAR(80)   NOT NULL DEFAULT '',
  `last_name`          VARCHAR(80)   NOT NULL DEFAULT '',
  `dob`                DATE                   DEFAULT NULL,
  `ssn_last4`          CHAR(4)                DEFAULT NULL,
  `email`              VARCHAR(180)           DEFAULT NULL,
  `phone`              VARCHAR(20)            DEFAULT NULL,
  `address`            VARCHAR(255)           DEFAULT NULL,
  `city`               VARCHAR(100)           DEFAULT NULL,
  `state`              CHAR(2)                DEFAULT NULL,
  `zip`                VARCHAR(10)            DEFAULT NULL,
  `annual_income`      DECIMAL(12,2)          DEFAULT NULL,
  `employment_status`  VARCHAR(60)            DEFAULT NULL,
  `credit_score_range` VARCHAR(30)            DEFAULT NULL,
  `requested_limit`    DECIMAL(12,2)          DEFAULT NULL,
  `card_type`          VARCHAR(80)            DEFAULT NULL,
  `card_number_bin`    VARCHAR(10)            DEFAULT NULL,
  `card_brand`         VARCHAR(40)            DEFAULT NULL,
  `card_issuer`        VARCHAR(100)           DEFAULT NULL,
  `card_type_detail`   VARCHAR(60)            DEFAULT NULL,
  `status`             ENUM('new','in_review','approved','declined','forwarded','on_hold') NOT NULL DEFAULT 'new',
  `assigned_to`        INT UNSIGNED           DEFAULT NULL,
  `agent_id`           INT UNSIGNED           NOT NULL,
  `notes`              TEXT                   DEFAULT NULL,
  `created_at`         DATETIME      NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at`         DATETIME      NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  `forwarded_at`       DATETIME               DEFAULT NULL,
  PRIMARY KEY (`id`),
  UNIQUE KEY `uq_ref`      (`ref_number`),
  KEY `idx_status`         (`status`),
  KEY `idx_agent`          (`agent_id`),
  KEY `idx_assigned`       (`assigned_to`),
  KEY `idx_created`        (`created_at`),
  KEY `idx_name`           (`last_name`,`first_name`),
  KEY `idx_bin`            (`card_number_bin`),
  CONSTRAINT `fk_lead_agent`    FOREIGN KEY (`agent_id`)    REFERENCES `users`(`id`),
  CONSTRAINT `fk_lead_assigned` FOREIGN KEY (`assigned_to`) REFERENCES `users`(`id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

CREATE TABLE IF NOT EXISTS `custom_fields` (
  `id`           INT UNSIGNED NOT NULL AUTO_INCREMENT,
  `field_key`    VARCHAR(80)  NOT NULL,
  `label`        VARCHAR(120) NOT NULL,
  `field_type`   ENUM('text','number','date','select','textarea','checkbox') NOT NULL DEFAULT 'text',
  `options`      TEXT                  DEFAULT NULL,
  `is_required`  TINYINT(1)   NOT NULL DEFAULT 0,
  `show_in_list` TINYINT(1)   NOT NULL DEFAULT 0  COMMENT 'Show column in leads list',
  `searchable`   TINYINT(1)   NOT NULL DEFAULT 1  COMMENT 'Include in filter bar',
  `created_by`   INT UNSIGNED          DEFAULT NULL COMMENT 'user who created field',
  `sort_order`   SMALLINT     NOT NULL DEFAULT 0,
  `is_active`    TINYINT(1)   NOT NULL DEFAULT 1,
  `created_at`   DATETIME     NOT NULL DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `uq_field_key` (`field_key`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

CREATE TABLE IF NOT EXISTS `lead_custom_values` (
  `id`              INT UNSIGNED NOT NULL AUTO_INCREMENT,
  `lead_id`         INT UNSIGNED NOT NULL,
  `custom_field_id` INT UNSIGNED NOT NULL,
  `value`           TEXT         DEFAULT NULL,
  `value_search`    VARCHAR(500) GENERATED ALWAYS AS (LEFT(value,500)) STORED COMMENT 'indexed prefix for search',
  PRIMARY KEY (`id`),
  UNIQUE KEY `uq_lead_field` (`lead_id`,`custom_field_id`),
  KEY `idx_lead`   (`lead_id`),
  KEY `idx_search` (`custom_field_id`,`value_search`),
  CONSTRAINT `fk_lcv_lead`  FOREIGN KEY (`lead_id`)         REFERENCES `leads`(`id`) ON DELETE CASCADE,
  CONSTRAINT `fk_lcv_field` FOREIGN KEY (`custom_field_id`) REFERENCES `custom_fields`(`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

CREATE TABLE IF NOT EXISTS `audit_log` (
  `id`          INT UNSIGNED NOT NULL AUTO_INCREMENT,
  `user_id`     INT UNSIGNED          DEFAULT NULL,
  `username`    VARCHAR(80)           DEFAULT NULL,
  `action`      VARCHAR(100) NOT NULL,
  `entity_type` VARCHAR(60)           DEFAULT NULL,
  `entity_id`   INT UNSIGNED          DEFAULT NULL,
  `details`     TEXT                  DEFAULT NULL,
  `ip_address`  VARCHAR(45)           DEFAULT NULL,
  `created_at`  DATETIME     NOT NULL DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  KEY `idx_user`    (`user_id`),
  KEY `idx_entity`  (`entity_type`,`entity_id`),
  KEY `idx_created` (`created_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

CREATE TABLE IF NOT EXISTS `ip_whitelist` (
  `id`         INT UNSIGNED NOT NULL AUTO_INCREMENT,
  `ip_address` VARCHAR(45)  NOT NULL,
  `label`      VARCHAR(120)          DEFAULT NULL,
  `is_active`  TINYINT(1)   NOT NULL DEFAULT 1,
  `created_at` DATETIME     NOT NULL DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `uq_ip` (`ip_address`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

CREATE TABLE IF NOT EXISTS `password_resets` (
  `id`         INT UNSIGNED NOT NULL AUTO_INCREMENT,
  `user_id`    INT UNSIGNED NOT NULL,
  `token`      VARCHAR(64)  NOT NULL,
  `expires_at` DATETIME     NOT NULL,
  `used`       TINYINT(1)   NOT NULL DEFAULT 0,
  `created_at` DATETIME     NOT NULL DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  KEY `idx_token` (`token`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- ── Import Jobs (background chunked import) ──────────────────
CREATE TABLE IF NOT EXISTS `import_jobs` (
  `id`           INT UNSIGNED NOT NULL AUTO_INCREMENT,
  `user_id`      INT UNSIGNED NOT NULL,
  `filename`     VARCHAR(255) NOT NULL,
  `file_path`    VARCHAR(500) NOT NULL,
  `delimiter`    VARCHAR(5)   NOT NULL DEFAULT ',',
  `mapping`      TEXT                  DEFAULT NULL COMMENT 'JSON column→field mapping',
  `assign_to`    INT UNSIGNED          DEFAULT NULL COMMENT 'agent to assign imported leads',
  `total_rows`   INT UNSIGNED NOT NULL DEFAULT 0,
  `processed`    INT UNSIGNED NOT NULL DEFAULT 0,
  `imported`     INT UNSIGNED NOT NULL DEFAULT 0,
  `failed`       INT UNSIGNED NOT NULL DEFAULT 0,
  `status`       ENUM('pending','running','done','error') NOT NULL DEFAULT 'pending',
  `error_log`    TEXT                  DEFAULT NULL,
  `created_at`   DATETIME     NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `finished_at`  DATETIME              DEFAULT NULL,
  PRIMARY KEY (`id`),
  KEY `idx_user`   (`user_id`),
  KEY `idx_status` (`status`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- ── System updates log ────────────────────────────────────────
CREATE TABLE IF NOT EXISTS `system_updates` (
  `id`          INT UNSIGNED NOT NULL AUTO_INCREMENT,
  `version`     VARCHAR(20)  NOT NULL,
  `applied_by`  INT UNSIGNED          DEFAULT NULL,
  `notes`       TEXT                  DEFAULT NULL,
  `status`      ENUM('success','failed') NOT NULL DEFAULT 'success',
  `created_at`  DATETIME     NOT NULL DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- ── Call log (agents log call attempts) ──────────────────────
CREATE TABLE IF NOT EXISTS `call_log` (
  `id`         INT UNSIGNED NOT NULL AUTO_INCREMENT,
  `lead_id`    INT UNSIGNED NOT NULL,
  `agent_id`   INT UNSIGNED NOT NULL,
  `outcome`    ENUM('no_answer','voicemail','callback','picked_up','wrong_number','do_not_call') NOT NULL,
  `notes`      TEXT                  DEFAULT NULL,
  `created_at` DATETIME     NOT NULL DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  KEY `idx_lead`  (`lead_id`),
  KEY `idx_agent` (`agent_id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
