<?php
require_once __DIR__.'/../../config/config.php';
require_once __DIR__.'/../../includes/db.php';
require_once __DIR__.'/../../includes/auth.php';
require_once __DIR__.'/../../includes/helpers.php';
session_start_secure();
require_login(); // ALL roles can access

$user   = current_user();
$db     = db();
$errors = [];

// Only admin can delete or deactivate fields created by others
function cf_can_edit(array $cf): bool {
    $u = current_user();
    return $u['role'] === 'admin' || (int)$cf['created_by'] === (int)$u['id'];
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    csrf_check();
    $action = $_POST['action'] ?? '';

    if ($action === 'save') {
        $fid       = (int)($_POST['field_id'] ?? 0);
        $label     = trim($_POST['label']     ?? '');
        $type      = $_POST['field_type']     ?? 'text';
        $rawKey    = trim($_POST['field_key'] ?? $label);
        $key       = preg_replace('/[^a-z0-9_]/','',strtolower(str_replace([' ','-'],'_',$rawKey)));
        $required  = isset($_POST['is_required'])  ? 1 : 0;
        $active    = isset($_POST['is_active'])    ? 1 : 0;
        $showList  = isset($_POST['show_in_list']) ? 1 : 0;
        $searchable= isset($_POST['searchable'])   ? 1 : 0;
        $opts      = trim($_POST['options'] ?? '');

        if (!$label) $errors[] = 'Label is required.';
        if (!$key)   $errors[] = 'Field key is required.';

        // Check for duplicate key (on new fields)
        if (!$errors && !$fid) {
            $dup = $db->prepare("SELECT id FROM custom_fields WHERE field_key=?");
            $dup->execute([$key]);
            if ($dup->fetch()) $errors[] = "Field key '$key' already exists. Choose a different name.";
        }

        if (!$errors) {
            $optJson = null;
            if ($type === 'select' && $opts) {
                $optArr  = array_values(array_filter(array_map('trim', explode("\n", $opts))));
                $optJson = json_encode($optArr);
            }
            if ($fid) {
                // Verify edit permission
                $existing = $db->prepare("SELECT * FROM custom_fields WHERE id=?");
                $existing->execute([$fid]);
                $existing = $existing->fetch();
                if (!$existing || !cf_can_edit($existing)) {
                    flash('error','You do not have permission to edit this field.');
                    redirect(APP_URL.'/modules/admin/custom_fields.php');
                }
                $db->prepare("UPDATE custom_fields SET label=?,field_type=?,is_required=?,is_active=?,
                    show_in_list=?,searchable=?,options=? WHERE id=?")
                   ->execute([$label,$type,$required,$active,$showList,$searchable,$optJson,$fid]);
                audit('field_updated','custom_field',$fid,"Key: $key");
                flash('success','Field updated.');
            } else {
                $maxOrder = (int)$db->query("SELECT MAX(sort_order) FROM custom_fields")->fetchColumn();
                $db->prepare("INSERT INTO custom_fields
                    (field_key,label,field_type,is_required,is_active,show_in_list,searchable,options,sort_order,created_by)
                    VALUES (?,?,?,?,?,?,?,?,?,?)")
                   ->execute([$key,$label,$type,$required,$active,$showList,$searchable,$optJson,$maxOrder+1,$user['id']]);
                $newId = (int)$db->lastInsertId();
                audit('field_created','custom_field',$newId,"Key: $key by {$user['username']}");
                flash('success','Custom field "'.$label.'" created. It is now visible on all lead forms.');
            }
            bust_cf_cache();
            redirect(APP_URL.'/modules/admin/custom_fields.php');
        }
    }

    if ($action === 'delete') {
        $fid = (int)($_POST['field_id'] ?? 0);
        $existing = $db->prepare("SELECT * FROM custom_fields WHERE id=?");
        $existing->execute([$fid]);
        $existing = $existing->fetch();
        if ($existing && cf_can_edit($existing)) {
            $db->prepare("DELETE FROM custom_fields WHERE id=?")->execute([$fid]);
            audit('field_deleted','custom_field',$fid,'Key: '.$existing['field_key']);
            flash('success','Field deleted.');
        } else {
            flash('error','Permission denied or field not found.');
        }
        bust_cf_cache();
        redirect(APP_URL.'/modules/admin/custom_fields.php');
    }

    if ($action === 'toggle_active') {
        $fid = (int)($_POST['field_id'] ?? 0);
        $existing = $db->prepare("SELECT * FROM custom_fields WHERE id=?");
        $existing->execute([$fid]);
        $existing = $existing->fetch();
        if ($existing && cf_can_edit($existing)) {
            $new = $existing['is_active'] ? 0 : 1;
            $db->prepare("UPDATE custom_fields SET is_active=? WHERE id=?")->execute([$new,$fid]);
            flash('success','Field '.($new?'activated':'deactivated').'.');
            bust_cf_cache();
        }
        redirect(APP_URL.'/modules/admin/custom_fields.php');
    }

    if ($action === 'reorder') {
        $order = json_decode($_POST['order'] ?? '[]', true);
        foreach ($order as $i => $fid) {
            $db->prepare("UPDATE custom_fields SET sort_order=? WHERE id=?")->execute([$i, (int)$fid]);
        }
        header('Content-Type: application/json');
        echo json_encode(['ok' => true]);
        exit;
    }
}

$fields = $db->query("SELECT cf.*, u.full_name AS creator_name, u.username AS creator_username
    FROM custom_fields cf LEFT JOIN users u ON u.id=cf.created_by ORDER BY cf.sort_order,cf.id")->fetchAll();

$pageTitle = 'Custom Fields';
include __DIR__.'/../../includes/layout_header.php';
?>

<div class="page-header d-flex align-items-start justify-content-between flex-wrap gap-2">
  <div>
    <h1><i class="bi bi-sliders me-2 text-primary fs-4"></i>Custom Fields</h1>
    <p class="text-muted small mb-0">
      Add and manage custom lead fields. Fields appear immediately on all lead forms for every user.<br>
      <?php if($user['role']!=='admin'): ?>
        <span class="text-warning"><i class="bi bi-info-circle me-1"></i>You can edit fields you created. Admins manage all fields.</span>
      <?php endif; ?>
    </p>
  </div>
  <button class="btn btn-primary btn-sm" data-bs-toggle="modal" data-bs-target="#fieldModal" id="addFieldBtn">
    <i class="bi bi-plus-lg me-1"></i>Add Field</button>
</div>

<?php if ($errors): ?>
<div class="alert alert-danger"><?= implode('<br>', $errors) ?></div>
<?php endif; ?>

<?php if (empty($fields)): ?>
<div class="card"><div class="card-body text-center py-5 text-muted">
  <i class="bi bi-sliders fs-2 d-block mb-2 opacity-25"></i>
  <div class="fw-medium">No custom fields yet.</div>
  <p class="small mt-1">Click "Add Field" to create your first custom field. It will immediately appear on all lead forms.</p>
  <button class="btn btn-primary btn-sm" data-bs-toggle="modal" data-bs-target="#fieldModal">Add First Field</button>
</div></div>
<?php else: ?>

<div class="card mb-3">
  <div class="table-responsive">
    <table class="table table-hover mb-0">
      <thead>
        <tr>
          <th style="width:32px" title="Drag to reorder"></th>
          <th>Label</th>
          <th>Key</th>
          <th>Type</th>
          <th title="Show as filter in leads list">Searchable</th>
          <th title="Show as column in leads list">List Column</th>
          <th>Required</th>
          <th>Active</th>
          <th>Created By</th>
          <th>Actions</th>
        </tr>
      </thead>
      <tbody id="sortableFields">
        <?php foreach ($fields as $f): ?>
        <tr data-id="<?= $f['id'] ?>" class="<?= !$f['is_active']?'opacity-50':'' ?>">
          <td>
            <?php if(cf_can_edit($f)||$user['role']==='admin'): ?>
            <i class="bi bi-grip-vertical sortable-handle text-muted"></i>
            <?php endif; ?>
          </td>
          <td class="fw-medium"><?= h($f['label']) ?></td>
          <td><code class="small"><?= h($f['field_key']) ?></code></td>
          <td><span class="tag"><?= h($f['field_type']) ?></span></td>
          <td class="text-center"><?= $f['searchable']?'<i class="bi bi-check-circle-fill text-success"></i>':'<span class="text-muted">—</span>' ?></td>
          <td class="text-center"><?= $f['show_in_list']?'<i class="bi bi-check-circle-fill text-info"></i>':'<span class="text-muted">—</span>' ?></td>
          <td class="text-center"><?= $f['is_required']?'<i class="bi bi-asterisk text-warning" style="font-size:.8rem"></i>':'<span class="text-muted">—</span>' ?></td>
          <td>
            <?php if(cf_can_edit($f)): ?>
            <form method="post" class="d-inline">
              <input type="hidden" name="csrf_token" value="<?= csrf_token() ?>">
              <input type="hidden" name="action" value="toggle_active">
              <input type="hidden" name="field_id" value="<?= $f['id'] ?>">
              <button type="submit" class="btn btn-link p-0 border-0" title="Toggle">
                <?= $f['is_active']
                  ? '<span class="status-badge badge-approved">Yes</span>'
                  : '<span class="status-badge badge-hold">No</span>' ?>
              </button>
            </form>
            <?php else: ?>
              <?= $f['is_active'] ? '<span class="status-badge badge-approved">Yes</span>' : '<span class="status-badge badge-hold">No</span>' ?>
            <?php endif; ?>
          </td>
          <td class="small text-muted"><?= h($f['creator_name']?:$f['creator_username']?:'System') ?></td>
          <td>
            <?php if(cf_can_edit($f)): ?>
            <button class="btn btn-sm btn-outline-primary p-1 lh-1"
              onclick="openEdit(<?= htmlspecialchars(json_encode($f)) ?>)" title="Edit">
              <i class="bi bi-pencil"></i></button>
            <form method="post" class="d-inline">
              <input type="hidden" name="csrf_token" value="<?= csrf_token() ?>">
              <input type="hidden" name="action" value="delete">
              <input type="hidden" name="field_id" value="<?= $f['id'] ?>">
              <button type="submit" class="btn btn-sm btn-outline-danger p-1 lh-1"
                data-confirm="Delete '<?= h($f['label']) ?>'? All stored values will be lost." title="Delete">
                <i class="bi bi-trash"></i></button>
            </form>
            <?php else: ?>
              <span class="text-muted small">Admin only</span>
            <?php endif; ?>
          </td>
        </tr>
        <?php endforeach; ?>
      </tbody>
    </table>
  </div>
</div>

<div class="alert alert-info small py-2">
  <i class="bi bi-lightbulb me-1"></i>
  <strong>Tips:</strong>
  Drag rows to reorder. Enable <strong>Searchable</strong> to show the field in the leads filter bar.
  Enable <strong>List Column</strong> to show the field as a column in the leads table.
  Fields marked inactive are hidden from forms but data is preserved.
</div>

<?php endif; ?>

<!-- Field Modal -->
<div class="modal fade" id="fieldModal" tabindex="-1">
  <div class="modal-dialog">
    <div class="modal-content">
      <form method="post" id="fieldForm">
        <input type="hidden" name="csrf_token" value="<?= csrf_token() ?>">
        <input type="hidden" name="action" value="save">
        <input type="hidden" name="field_id" id="modal_field_id" value="0">
        <div class="modal-header">
          <h5 class="modal-title" id="fieldModalTitle">Add Custom Field</h5>
          <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
        </div>
        <div class="modal-body">
          <div class="row g-3">
            <div class="col-md-6">
              <label class="form-label">Field Label *</label>
              <input type="text" name="label" id="modal_label" class="form-control" required
                     placeholder="e.g. Referral Source">
            </div>
            <div class="col-md-6">
              <label class="form-label">Field Key <span class="text-muted small">(auto-generated)</span></label>
              <input type="text" name="field_key" id="modal_field_key" class="form-control mono"
                     placeholder="referral_source" pattern="[a-z0-9_]+">
              <div class="form-text">Lowercase, underscores only. Cannot be changed after creation.</div>
            </div>
            <div class="col-md-6">
              <label class="form-label">Field Type</label>
              <select name="field_type" id="modal_field_type" class="form-select">
                <option value="text">Text (single line)</option>
                <option value="number">Number</option>
                <option value="date">Date</option>
                <option value="select">Dropdown (Select)</option>
                <option value="textarea">Text Area (multi-line)</option>
                <option value="checkbox">Checkbox (yes/no)</option>
              </select>
            </div>
            <div class="col-md-6">
              <!-- Checkboxes -->
              <label class="form-label d-block">Options</label>
              <div class="d-flex flex-column gap-2">
                <div class="form-check">
                  <input type="checkbox" name="is_required" id="modal_required" class="form-check-input" value="1">
                  <label class="form-check-label" for="modal_required">Required on lead forms</label>
                </div>
                <div class="form-check">
                  <input type="checkbox" name="searchable" id="modal_searchable" class="form-check-input" value="1" checked>
                  <label class="form-check-label" for="modal_searchable">Show in filter bar</label>
                </div>
                <div class="form-check">
                  <input type="checkbox" name="show_in_list" id="modal_show_list" class="form-check-input" value="1">
                  <label class="form-check-label" for="modal_show_list">Show as leads list column</label>
                </div>
                <div class="form-check">
                  <input type="checkbox" name="is_active" id="modal_active" class="form-check-input" value="1" checked>
                  <label class="form-check-label" for="modal_active">Active (visible on forms)</label>
                </div>
              </div>
            </div>
            <div class="col-12" id="optionsRow" style="display:none">
              <label class="form-label">Dropdown Options <span class="text-muted small">(one per line)</span></label>
              <textarea name="options" id="modal_options" class="form-control mono" rows="5"
                placeholder="Option A&#10;Option B&#10;Option C"></textarea>
            </div>
          </div>
        </div>
        <div class="modal-footer">
          <button type="button" class="btn btn-outline-secondary" data-bs-dismiss="modal">Cancel</button>
          <button type="submit" class="btn btn-primary">
            <i class="bi bi-check-lg me-1"></i>Save Field</button>
        </div>
      </form>
    </div>
  </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/sortablejs@1.15.2/Sortable.min.js"></script>
<script>
// Drag to reorder
const tbody = document.getElementById('sortableFields');
if (tbody) {
  Sortable.create(tbody, {
    handle: '.sortable-handle', animation: 120,
    onEnd: () => {
      const ids = [...tbody.querySelectorAll('tr')].map(r => r.dataset.id);
      fetch('<?= APP_URL ?>/modules/admin/custom_fields.php', {
        method: 'POST',
        headers: {'Content-Type':'application/x-www-form-urlencoded'},
        body: 'action=reorder&order='+JSON.stringify(ids)+'&csrf_token=<?= csrf_token() ?>'
      });
    }
  });
}

// Show/hide options textarea
const typeSelect = document.getElementById('modal_field_type');
if (typeSelect) {
  typeSelect.addEventListener('change', function () {
    document.getElementById('optionsRow').style.display = this.value === 'select' ? '' : 'none';
  });
}

// Auto-generate key from label
const labelInp = document.getElementById('modal_label');
const keyInp   = document.getElementById('modal_field_key');
labelInp?.addEventListener('input', function () {
  if (!keyInp.dataset.manual) {
    keyInp.value = this.value.toLowerCase()
      .replace(/[^a-z0-9]+/g,'_').replace(/^_+|_+$/g,'').slice(0,40);
  }
});
keyInp?.addEventListener('input', function () { this.dataset.manual = '1'; });

function openEdit(f) {
  document.getElementById('fieldModalTitle').textContent = 'Edit Field: ' + f.label;
  document.getElementById('modal_field_id').value = f.id;
  document.getElementById('modal_label').value = f.label;
  const k = document.getElementById('modal_field_key');
  k.value = f.field_key; k.dataset.manual = '1';
  k.readOnly = true; // Key cannot change after creation
  document.getElementById('modal_field_type').value = f.field_type;
  document.getElementById('modal_required').checked  = f.is_required == 1;
  document.getElementById('modal_searchable').checked = f.searchable == 1;
  document.getElementById('modal_show_list').checked = f.show_in_list == 1;
  document.getElementById('modal_active').checked    = f.is_active == 1;
  const optRow = document.getElementById('optionsRow');
  const optTA  = document.getElementById('modal_options');
  if (f.field_type === 'select') {
    optRow.style.display = '';
    try { optTA.value = (JSON.parse(f.options)||[]).join('\n'); } catch(e) { optTA.value=''; }
  } else { optRow.style.display = 'none'; optTA.value = ''; }
  new bootstrap.Modal(document.getElementById('fieldModal')).show();
}

// Reset for new field
document.getElementById('addFieldBtn')?.addEventListener('click', () => {
  document.getElementById('fieldModalTitle').textContent = 'Add Custom Field';
  document.getElementById('fieldForm').reset();
  document.getElementById('modal_field_id').value = '0';
  delete keyInp.dataset.manual;
  keyInp.readOnly = false;
  document.getElementById('optionsRow').style.display = 'none';
  document.getElementById('modal_active').checked     = true;
  document.getElementById('modal_searchable').checked = true;
});
</script>

<?php include __DIR__.'/../../includes/layout_footer.php'; ?>
