<?php
require_once __DIR__.'/../../config/config.php';
require_once __DIR__.'/../../includes/db.php';
require_once __DIR__.'/../../includes/auth.php';
require_once __DIR__.'/../../includes/helpers.php';
session_start_secure();
require_role('admin');

$user   = current_user();
$db     = db();
$errors = [];
$result = null;

// ── Current version (read from file if it exists) ──────────
$versionFile = __DIR__.'/../../VERSION';
$currentVersion = file_exists($versionFile) ? trim(file_get_contents($versionFile)) : 'v1.0.0';

// ── Handle update upload ───────────────────────────────────
if ($_SERVER['REQUEST_METHOD'] === 'POST' && !empty($_FILES['update_zip']['tmp_name'])) {
    csrf_check();

    $file = $_FILES['update_zip'];
    $ext  = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));

    if ($ext !== 'zip') {
        $errors[] = 'Update package must be a .zip file.';
    } elseif ($file['size'] > 50 * 1024 * 1024) {
        $errors[] = 'Update file too large (max 50MB).';
    } elseif (!extension_loaded('zip')) {
        $errors[] = 'PHP zip extension is not available on this server. Please contact your host.';
    } else {
        $tmpZip  = sys_get_temp_dir().'/crm_update_'.uniqid().'.zip';
        move_uploaded_file($file['tmp_name'], $tmpZip);

        // ── Step 1: inspect zip ────────────────────────────
        $zip = new ZipArchive();
        if ($zip->open($tmpZip) !== true) {
            $errors[] = 'Could not open ZIP file. It may be corrupted.';
        } else {
            // Find root prefix (some ZIPs have crm/ prefix, some don't)
            $firstEntry = $zip->getNameIndex(0);
            $prefix = '';
            if ($firstEntry && str_contains($firstEntry, '/')) {
                $parts  = explode('/', $firstEntry);
                $prefix = $parts[0].'/';
            }

            // Check for version file inside zip
            $newVersion = $currentVersion;
            $vInZip = $zip->locateName($prefix.'VERSION');
            if ($vInZip !== false) {
                $newVersion = trim($zip->getFromIndex($vInZip));
            }

            // ── Step 2: backup config ──────────────────────
            $configPath  = __DIR__.'/../../config/config.php';
            $configBackup = sys_get_temp_dir().'/crm_config_backup_'.uniqid().'.php';
            copy($configPath, $configBackup);

            // ── Step 3: extract to temp dir ───────────────
            $tmpExtract = sys_get_temp_dir().'/crm_extract_'.uniqid().'/';
            @mkdir($tmpExtract, 0755, true);
            $zip->extractTo($tmpExtract);
            $zip->close();

            // ── Step 4: copy files (skip config, install) ─
            $srcDir  = $tmpExtract . $prefix;
            $destDir = realpath(__DIR__.'/../../');
            $skipped = [];
            $copied  = 0;
            $failed  = 0;

            // Files/dirs to never overwrite during update
            $protect = ['config/config.php', 'install/'];

            $iterator = new RecursiveIteratorIterator(
                new RecursiveDirectoryIterator($srcDir, RecursiveDirectoryIterator::SKIP_DOTS),
                RecursiveIteratorIterator::SELF_FIRST
            );

            foreach ($iterator as $item) {
                $relative = str_replace($srcDir, '', $item->getPathname());
                $relative = ltrim($relative, '/\\');

                // Skip protected
                $skip = false;
                foreach ($protect as $p) {
                    if (str_starts_with($relative, $p)) { $skip=true; $skipped[]=$relative; break; }
                }
                if ($skip) continue;

                $dest = $destDir.'/'.$relative;

                if ($item->isDir()) {
                    @mkdir($dest, 0755, true);
                } else {
                    // Ensure parent dir exists
                    @mkdir(dirname($dest), 0755, true);
                    if (!copy($item->getPathname(), $dest)) {
                        $failed++;
                    } else {
                        $copied++;
                    }
                }
            }

            // ── Step 5: restore config ────────────────────
            copy($configBackup, $configPath);
            @unlink($configBackup);

            // ── Step 6: write new version ─────────────────
            if ($newVersion !== $currentVersion) {
                file_put_contents($versionFile, $newVersion);
            }

            // ── Step 7: run any pending SQL migrations ────
            $sqlRan = false; $sqlErrors = [];
            $migrDir = $destDir.'/install/migrations/';
            if (is_dir($migrDir)) {
                // Track applied migrations
                try {
                    $db->query("CREATE TABLE IF NOT EXISTS applied_migrations (
                        filename VARCHAR(255) PRIMARY KEY, applied_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP
                    )");
                    $applied = $db->query("SELECT filename FROM applied_migrations")->fetchAll(PDO::FETCH_COLUMN);
                    foreach (glob($migrDir.'*.sql') as $sqlFile) {
                        $fname = basename($sqlFile);
                        if (!in_array($fname, $applied)) {
                            try {
                                $sql = file_get_contents($sqlFile);
                                foreach (explode(';', $sql) as $stmt) {
                                    $stmt = trim($stmt);
                                    if ($stmt) $db->exec($stmt);
                                }
                                $db->prepare("INSERT IGNORE INTO applied_migrations (filename) VALUES (?)")->execute([$fname]);
                                $sqlRan = true;
                            } catch (Exception $e) {
                                $sqlErrors[] = "$fname: ".$e->getMessage();
                            }
                        }
                    }
                } catch (Exception $e) {
                    $sqlErrors[] = 'Migration check failed: '.$e->getMessage();
                }
            }

            // ── Cleanup ────────────────────────────────────
            @unlink($tmpZip);
            @array_map('unlink', glob($tmpExtract.'*'));
            @rmdir($tmpExtract);

            // ── Log ────────────────────────────────────────
            $details = "Updated to $newVersion. Files: $copied copied, $failed failed.".
                       ($sqlRan?' SQL migrations applied.':'').
                       ($sqlErrors?' SQL errors: '.implode('; ',$sqlErrors):'');
            $db->prepare("INSERT INTO system_updates (version,applied_by,notes,status) VALUES (?,?,?,?)")
               ->execute([$newVersion,$user['id'],$details,$failed?'failed':'success']);
            audit('system_update','system',null,$details);

            $result = [
                'version'   => $newVersion,
                'copied'    => $copied,
                'failed'    => $failed,
                'sql'       => $sqlRan,
                'sql_errors'=> $sqlErrors,
                'skipped'   => count($skipped),
            ];

            // Reload page after short delay so new code is served
            header("Refresh: 3; url=".APP_URL."/modules/admin/updater.php?updated=1");
        }
    }
}

// History
$history = $db->query("SELECT su.*,u.full_name,u.username FROM system_updates su LEFT JOIN users u ON u.id=su.applied_by ORDER BY su.created_at DESC LIMIT 10")->fetchAll();

$pageTitle = 'System Update';
include __DIR__.'/../../includes/layout_header.php';
?>

<div class="page-header d-flex align-items-start justify-content-between flex-wrap gap-2">
  <div>
    <h1><i class="bi bi-arrow-repeat me-2 text-primary fs-4"></i>System Update</h1>
    <p class="text-muted small mb-0">
      Running <strong><?= h(APP_NAME) ?></strong> version
      <code><?= h($currentVersion) ?></code>
    </p>
  </div>
</div>

<?php if(!empty($_GET['updated'])): ?>
<div class="alert alert-success">
  <i class="bi bi-check-circle-fill me-2"></i>
  <strong>Update applied successfully.</strong> The page has been reloaded with the new code.
</div>
<?php endif; ?>

<?php if($errors): ?>
<div class="alert alert-danger">
  <i class="bi bi-exclamation-triangle me-2"></i>
  <?= implode('<br>',$errors) ?>
</div>
<?php endif; ?>

<?php if($result): ?>
<div class="alert alert-<?= $result['failed']?'warning':'success' ?>">
  <strong><i class="bi bi-check2-all me-2"></i>Update complete — version <?= h($result['version']) ?></strong><br>
  <span class="small">
    <?= $result['copied'] ?> files updated.
    <?php if($result['failed']): ?><span class="text-danger"><?= $result['failed'] ?> failed.</span><?php endif; ?>
    <?php if($result['sql']): ?>SQL migrations applied.<?php endif; ?>
    <?php if($result['skipped']): ?><?= $result['skipped'] ?> protected files skipped (config preserved).<?php endif; ?>
    <?php if($result['sql_errors']): ?>
      <br><span class="text-danger">SQL errors: <?= h(implode('; ',$result['sql_errors'])) ?></span>
    <?php endif; ?>
  </span>
  <br><small class="text-muted">Reloading in 3 seconds…</small>
</div>
<?php endif; ?>

<div class="row g-4">
  <!-- Upload card -->
  <div class="col-lg-7">
    <div class="card">
      <div class="card-header fw-semibold">
        <i class="bi bi-upload me-2 text-primary"></i>Upload Update Package
      </div>
      <div class="card-body">
        <form method="post" enctype="multipart/form-data" id="updateForm">
          <input type="hidden" name="csrf_token" value="<?= csrf_token() ?>">

          <div class="alert alert-info small py-2 mb-3">
            <i class="bi bi-info-circle me-1"></i>
            Upload the <code>.zip</code> file provided by Claude (or exported from the CRM build).
            Your <strong>database and config are automatically preserved</strong> — only PHP/JS/CSS files are replaced.
          </div>

          <div class="drop-zone mb-3" id="dropZone">
            <i class="bi bi-file-earmark-zip d-block mb-2 fs-2 opacity-40"></i>
            <div class="fw-medium mb-1">Drag & drop update ZIP here</div>
            <div class="text-muted small mb-3">or click to browse</div>
            <input type="file" name="update_zip" id="zipInput" accept=".zip" class="d-none" required>
            <button type="button" class="btn btn-outline-primary btn-sm"
              onclick="document.getElementById('zipInput').click()">
              <i class="bi bi-folder2-open me-1"></i>Choose ZIP</button>
            <div id="zipName" class="mt-2 small fw-semibold text-primary"></div>
          </div>

          <div class="alert alert-warning py-2 small mb-3">
            <i class="bi bi-exclamation-triangle me-1"></i>
            <strong>Important:</strong> Always back up your database before updating.
            The update will overwrite all application files except <code>config/config.php</code>
            and the <code>install/</code> folder.
          </div>

          <button type="submit" class="btn btn-primary w-100" id="updateBtn" disabled
            onclick="return confirm('Apply this update? Files will be overwritten. Your config and database are safe.\n\nContinue?')">
            <i class="bi bi-lightning-charge me-1"></i>Apply Update Now</button>
        </form>
      </div>
    </div>

    <!-- How to get an update -->
    <div class="card mt-4">
      <div class="card-header fw-semibold">
        <i class="bi bi-question-circle me-2 text-primary"></i>How to Get a New Version
      </div>
      <div class="card-body">
        <ol class="mb-0 small" style="line-height:2">
          <li>Open a new chat with <strong>Claude</strong> (claude.ai)</li>
          <li>Paste this prompt:<br>
            <code class="d-block p-2 rounded my-2" style="background:var(--bs-tertiary-bg);font-size:.8rem;user-select:all"
              id="promptText">I have the Financial Leads CRM at version <?= h($currentVersion) ?>. Please generate an updated complete ZIP package with [describe your changes here].</code>
            <button class="btn btn-xs btn-outline-secondary btn-sm py-0 px-2" onclick="copyPrompt()">
              <i class="bi bi-clipboard me-1"></i>Copy</button>
          </li>
          <li>Download the <code>.zip</code> Claude provides</li>
          <li>Upload it above and click <strong>Apply Update</strong></li>
        </ol>
      </div>
    </div>
  </div>

  <!-- Update history -->
  <div class="col-lg-5">
    <div class="card">
      <div class="card-header fw-semibold">
        <i class="bi bi-clock-history me-2 text-primary"></i>Update History
      </div>
      <?php if($history): ?>
      <div class="table-responsive">
        <table class="table table-sm mb-0">
          <thead><tr><th>Version</th><th>Status</th><th>By</th><th>When</th></tr></thead>
          <tbody>
            <?php foreach($history as $h): ?>
            <tr>
              <td class="mono small fw-medium"><?= htmlspecialchars($h['version']) ?></td>
              <td>
                <span class="badge <?= $h['status']==='success'?'bg-success':'bg-danger' ?>">
                  <?= $h['status'] ?></span>
              </td>
              <td class="small text-muted"><?= htmlspecialchars($h['full_name']?:$h['username']?:'—') ?></td>
              <td class="small text-muted"><?= htmlspecialchars(date('M j Y', strtotime($h['created_at']))) ?></td>
            </tr>
            <?php endforeach; ?>
          </tbody>
        </table>
      </div>
      <?php else: ?>
      <div class="card-body text-center text-muted py-4">
        <i class="bi bi-clock-history opacity-25 fs-3 d-block mb-1"></i>No updates applied yet.
      </div>
      <?php endif; ?>
    </div>

    <!-- Server info -->
    <div class="card mt-4">
      <div class="card-header fw-semibold">
        <i class="bi bi-server me-2 text-primary"></i>Server Info
      </div>
      <div class="card-body">
        <table class="table table-sm mb-0 small">
          <tr><td class="text-muted">PHP</td><td class="mono"><?= phpversion() ?></td></tr>
          <tr><td class="text-muted">ZIP Extension</td>
            <td><?= extension_loaded('zip') ?
              '<span class="text-success"><i class="bi bi-check-circle"></i> Available</span>' :
              '<span class="text-danger"><i class="bi bi-x-circle"></i> Not available</span>' ?></td></tr>
          <tr><td class="text-muted">App root</td><td class="mono small"><?= h(realpath(__DIR__.'/../../')) ?></td></tr>
          <tr><td class="text-muted">Writable</td>
            <td><?= is_writable(realpath(__DIR__.'/../../')) ?
              '<span class="text-success"><i class="bi bi-check-circle"></i> Yes</span>' :
              '<span class="text-danger"><i class="bi bi-x-circle"></i> No — check permissions</span>' ?></td></tr>
          <tr><td class="text-muted">Max upload</td><td class="mono"><?= ini_get('upload_max_filesize') ?></td></tr>
        </table>
      </div>
    </div>
  </div>
</div>

<script>
const zi=document.getElementById('zipInput');
const zn=document.getElementById('zipName');
const ub=document.getElementById('updateBtn');
const dz=document.getElementById('dropZone');
if(zi){
  zi.addEventListener('change',()=>{zn.textContent=zi.files[0]?.name||'';ub.disabled=!zi.files[0];});
}
if(dz){
  dz.addEventListener('dragover',e=>{e.preventDefault();dz.classList.add('dragover');});
  dz.addEventListener('dragleave',()=>dz.classList.remove('dragover'));
  dz.addEventListener('drop',e=>{
    e.preventDefault();dz.classList.remove('dragover');
    const f=e.dataTransfer.files[0]; if(!f)return;
    const dt=new DataTransfer();dt.items.add(f);zi.files=dt.files;
    zn.textContent=f.name; ub.disabled=false;
  });
  dz.addEventListener('click',e=>{if(!e.target.closest('button'))zi.click();});
}
function copyPrompt(){
  const t=document.getElementById('promptText').textContent;
  navigator.clipboard.writeText(t).then(()=>{
    const b=event.target.closest('button');
    const orig=b.innerHTML; b.innerHTML='<i class="bi bi-check me-1"></i>Copied!';
    setTimeout(()=>b.innerHTML=orig,2000);
  });
}
</script>

<?php include __DIR__.'/../../includes/layout_footer.php'; ?>
