<?php
require_once __DIR__.'/../../config/config.php';
require_once __DIR__.'/../../includes/db.php';
require_once __DIR__.'/../../includes/auth.php';
require_once __DIR__.'/../../includes/helpers.php';
session_start_secure();
require_login();

$user = current_user();
$db   = db();
$customFields = get_custom_fields();

$errors = [];
$lead   = [
    'first_name'=>'','last_name'=>'','email'=>'','phone'=>'','dob'=>'',
    'ssn_last4'=>'','address'=>'','city'=>'','state'=>'','zip'=>'',
    'annual_income'=>'','employment_status'=>'','credit_score_range'=>'',
    'requested_limit'=>'','card_type'=>'','card_number_bin'=>'',
    'card_brand'=>'','card_issuer'=>'','card_type_detail'=>'',
    'status'=>'new','assigned_to'=>'','notes'=>'',
];

if ($_SERVER['REQUEST_METHOD']==='POST') {
    csrf_check();

    $lead = array_merge($lead, array_map('trim',[
        'first_name'         => $_POST['first_name']??'',
        'last_name'          => $_POST['last_name']??'',
        'email'              => $_POST['email']??'',
        'phone'              => preg_replace('/\D/','', $_POST['phone']??''),
        'dob'                => $_POST['dob']??'',
        'ssn_last4'          => preg_replace('/\D/','', $_POST['ssn_last4']??''),
        'address'            => $_POST['address']??'',
        'city'               => $_POST['city']??'',
        'state'              => $_POST['state']??'',
        'zip'                => $_POST['zip']??'',
        'annual_income'      => $_POST['annual_income']??'',
        'employment_status'  => $_POST['employment_status']??'',
        'credit_score_range' => $_POST['credit_score_range']??'',
        'requested_limit'    => $_POST['requested_limit']??'',
        'card_type'          => $_POST['card_type']??'',
        'card_number_bin'    => preg_replace('/\D/','', $_POST['card_number_bin']??''),
        'card_brand'         => $_POST['card_brand']??'',
        'card_issuer'        => $_POST['card_issuer']??'',
        'card_type_detail'   => $_POST['card_type_detail']??'',
        'status'             => $_POST['status']??'new',
        'assigned_to'        => $_POST['assigned_to']??'',
        'notes'              => $_POST['notes']??'',
    ]));

    // Validate
    if (!$lead['first_name']) $errors[] = 'First name is required.';
    if (!$lead['last_name'])  $errors[] = 'Last name is required.';
    if ($lead['email'] && !filter_var($lead['email'],FILTER_VALIDATE_EMAIL)) $errors[] = 'Invalid email address.';
    if ($lead['ssn_last4'] && strlen($lead['ssn_last4'])!==4) $errors[] = 'SSN last 4 must be exactly 4 digits.';
    foreach($customFields as $cf) {
        if ($cf['is_required'] && empty($_POST['cf_'.$cf['field_key']])) {
            $errors[] = h($cf['label']).' is required.';
        }
    }

    if (!$errors) {
        $ref = generate_ref();
        $stmt = $db->prepare("INSERT INTO leads
            (ref_number,first_name,last_name,email,phone,dob,ssn_last4,address,city,state,zip,
             annual_income,employment_status,credit_score_range,requested_limit,card_type,
             card_number_bin,card_brand,card_issuer,card_type_detail,
             status,assigned_to,agent_id,notes,forwarded_at)
            VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)
        ");
        $forwarded_at = ($lead['status']==='forwarded') ? date('Y-m-d H:i:s') : null;
        $stmt->execute([
            $ref,$lead['first_name'],$lead['last_name'],$lead['email']?:null,$lead['phone']?:null,
            $lead['dob']?:null,$lead['ssn_last4']?:null,$lead['address']?:null,
            $lead['city']?:null,$lead['state']?:null,$lead['zip']?:null,
            $lead['annual_income']?:null,$lead['employment_status']?:null,
            $lead['credit_score_range']?:null,$lead['requested_limit']?:null,
            $lead['card_type']?:null,$lead['card_number_bin']?:null,
            $lead['card_brand']?:null,$lead['card_issuer']?:null,$lead['card_type_detail']?:null,
            $lead['status'],$lead['assigned_to']?:null,$user['id'],
            $lead['notes']?:null,$forwarded_at,
        ]);
        $newId = (int)$db->lastInsertId();
        save_custom_values($newId, $customFields, $_POST);
        audit('lead_created','lead',$newId,"Ref: $ref");
        flash('success',"Lead $ref created successfully.");

        if (!empty($_POST['save_forward'])) {
            $db->prepare("UPDATE leads SET status='forwarded',forwarded_at=NOW() WHERE id=?")->execute([$newId]);
            audit('lead_forwarded','lead',$newId,"Ref: $ref");
            flash('success',"Lead $ref created and forwarded to processing.");
        }
        redirect(APP_URL.'/modules/leads/view.php?id='.$newId);
    }
}

$agents    = ($user['role']!=='agent') ? $db->query("SELECT id,full_name,username FROM users WHERE role IN ('agent','admin') AND is_active=1 ORDER BY full_name")->fetchAll() : [];
$processors = $db->query("SELECT id,full_name,username FROM users WHERE role IN ('processor','admin') AND is_active=1 ORDER BY full_name")->fetchAll();

$pageTitle = 'Add New Lead';
include __DIR__.'/../../includes/layout_header.php';
?>

<div class="page-header">
  <h1><i class="bi bi-person-plus me-2 text-primary fs-4"></i>Add New Lead</h1>
</div>

<?php if($errors): ?>
<div class="alert alert-danger">
  <i class="bi bi-exclamation-triangle me-2"></i><strong>Please fix the following errors:</strong>
  <ul class="mb-0 mt-1"><?php foreach($errors as $e): ?><li><?= $e ?></li><?php endforeach; ?></ul>
</div>
<?php endif; ?>

<form method="post" id="leadForm" autocomplete="off">
  <input type="hidden" name="csrf_token" value="<?= csrf_token() ?>">

  <div class="row g-4">
    <!-- Left col -->
    <div class="col-lg-8">

      <!-- Applicant Info -->
      <div class="card mb-4">
        <div class="card-body">
          <p class="form-section-title">Applicant Information</p>
          <div class="row g-3">
            <div class="col-md-4">
              <label class="form-label">First Name *</label>
              <input type="text" name="first_name" class="form-control" data-fmt="capitalize"
                     value="<?= h($lead['first_name']) ?>" required>
            </div>
            <div class="col-md-4">
              <label class="form-label">Last Name *</label>
              <input type="text" name="last_name" class="form-control" data-fmt="capitalize"
                     value="<?= h($lead['last_name']) ?>" required>
            </div>
            <div class="col-md-4">
              <label class="form-label">Date of Birth</label>
              <input type="date" name="dob" class="form-control" value="<?= h($lead['dob']) ?>">
            </div>
            <div class="col-md-6">
              <label class="form-label">Email</label>
              <input type="email" name="email" class="form-control" value="<?= h($lead['email']) ?>">
            </div>
            <div class="col-md-6">
              <label class="form-label">Phone</label>
              <input type="text" name="phone" class="form-control" data-fmt="phone"
                     value="<?= h($lead['phone']?format_phone($lead['phone']):'') ?>" placeholder="(555) 000-0000">
            </div>
            <div class="col-md-4">
              <label class="form-label">SSN Last 4</label>
              <input type="text" name="ssn_last4" class="form-control mono" maxlength="4" placeholder="0000"
                     value="<?= h($lead['ssn_last4']) ?>">
            </div>
          </div>
        </div>
      </div>

      <!-- Address -->
      <div class="card mb-4">
        <div class="card-body">
          <p class="form-section-title">Address</p>
          <div class="row g-3">
            <div class="col-12">
              <label class="form-label">Street Address</label>
              <input type="text" name="address" class="form-control" value="<?= h($lead['address']) ?>">
            </div>
            <div class="col-md-4">
              <label class="form-label">ZIP Code</label>
              <input type="text" name="zip" class="form-control" maxlength="10" value="<?= h($lead['zip']) ?>" placeholder="Auto-fills city/state">
            </div>
            <div class="col-md-5">
              <label class="form-label">City</label>
              <input type="text" name="city" class="form-control" value="<?= h($lead['city']) ?>">
            </div>
            <div class="col-md-3">
              <label class="form-label">State</label>
              <select name="state" class="form-select">
                <option value="">—</option>
                <?php foreach(us_states() as $s): ?>
                  <option value="<?= $s ?>" <?= $lead['state']===$s?'selected':'' ?>><?= $s ?></option>
                <?php endforeach; ?>
              </select>
            </div>
          </div>
        </div>
      </div>

      <!-- Financial Info -->
      <div class="card mb-4">
        <div class="card-body">
          <p class="form-section-title">Financial Information</p>
          <div class="row g-3">
            <div class="col-md-4">
              <label class="form-label">Annual Income</label>
              <div class="input-group">
                <span class="input-group-text">$</span>
                <input type="number" name="annual_income" class="form-control" min="0" step="1000"
                       value="<?= h($lead['annual_income']) ?>" placeholder="0">
              </div>
            </div>
            <div class="col-md-4">
              <label class="form-label">Employment Status</label>
              <select name="employment_status" class="form-select">
                <option value="">— Select —</option>
                <?php foreach(['Employed','Self-Employed','Unemployed','Retired','Student','Part-Time'] as $e): ?>
                  <option value="<?= $e ?>" <?= $lead['employment_status']===$e?'selected':'' ?>><?= $e ?></option>
                <?php endforeach; ?>
              </select>
            </div>
            <div class="col-md-4">
              <label class="form-label">Credit Score Range</label>
              <select name="credit_score_range" class="form-select">
                <option value="">— Select —</option>
                <?php foreach(['300-579 (Poor)','580-669 (Fair)','670-739 (Good)','740-799 (Very Good)','800-850 (Exceptional)'] as $r): ?>
                  <option value="<?= $r ?>" <?= $lead['credit_score_range']===$r?'selected':'' ?>><?= $r ?></option>
                <?php endforeach; ?>
              </select>
            </div>
            <div class="col-md-4">
              <label class="form-label">Requested Credit Limit</label>
              <div class="input-group">
                <span class="input-group-text">$</span>
                <input type="number" name="requested_limit" class="form-control" min="0" step="500"
                       value="<?= h($lead['requested_limit']) ?>" placeholder="0">
              </div>
            </div>
            <div class="col-md-4">
              <label class="form-label">Card Type Preference</label>
              <select name="card_type" class="form-select">
                <option value="">— Select —</option>
                <?php foreach(['Visa','Mastercard','American Express','Discover','Any'] as $ct): ?>
                  <option value="<?= $ct ?>" <?= $lead['card_type']===$ct?'selected':'' ?>><?= $ct ?></option>
                <?php endforeach; ?>
              </select>
            </div>
          </div>
        </div>
      </div>

      <!-- BIN / Card Info -->
      <div class="card mb-4">
        <div class="card-body">
          <p class="form-section-title">Card / BIN Information</p>
          <div class="row g-3">
            <div class="col-md-5">
              <label class="form-label">
                BIN Number
                <span class="text-muted small">(first 6–8 digits)</span>
              </label>
              <input type="text" name="card_number_bin" id="card_number_bin" class="form-control mono"
                     maxlength="16" value="<?= h($lead['card_number_bin']) ?>" placeholder="e.g. 411111">
              <div class="mt-2" id="bin_status"></div>
            </div>
            <div class="col-md-3">
              <label class="form-label">Brand</label>
              <input type="text" name="card_brand" id="card_brand" class="form-control"
                     value="<?= h($lead['card_brand']) ?>" placeholder="Auto-filled">
            </div>
            <div class="col-md-4">
              <label class="form-label">Issuer / Bank</label>
              <input type="text" name="card_issuer" id="card_issuer" class="form-control"
                     value="<?= h($lead['card_issuer']) ?>" placeholder="Auto-filled">
            </div>
            <div class="col-md-4">
              <label class="form-label">Card Type Detail</label>
              <input type="text" name="card_type_detail" id="card_type_detail" class="form-control"
                     value="<?= h($lead['card_type_detail']) ?>" placeholder="e.g. debit, credit">
            </div>
          </div>
        </div>
      </div>

      <!-- Custom Fields -->
      <?php if($customFields): ?>
      <div class="card mb-4">
        <div class="card-body">
          <p class="form-section-title">Additional Fields</p>
          <div class="row g-3">
            <?php foreach($customFields as $cf):
              $cfVal = $lead['cf_'.$cf['field_key']] ?? '';
            ?>
            <div class="col-md-<?= in_array($cf['field_type'],['textarea']) ? '12':'6' ?>">
              <label class="form-label"><?= h($cf['label']) ?><?= $cf['is_required']?' *':'' ?></label>
              <?php if($cf['field_type']==='textarea'): ?>
                <textarea name="cf_<?= h($cf['field_key']) ?>" class="form-control" rows="3"
                  <?= $cf['is_required']?'required':'' ?>><?= h($cfVal) ?></textarea>
              <?php elseif($cf['field_type']==='select'): ?>
                <select name="cf_<?= h($cf['field_key']) ?>" class="form-select" <?= $cf['is_required']?'required':'' ?>>
                  <option value="">— Select —</option>
                  <?php $opts=json_decode($cf['options']??'[]',true);
                  foreach($opts as $o): ?>
                    <option value="<?= h($o) ?>" <?= $cfVal===$o?'selected':'' ?>><?= h($o) ?></option>
                  <?php endforeach; ?>
                </select>
              <?php elseif($cf['field_type']==='checkbox'): ?>
                <div class="form-check mt-2">
                  <input type="checkbox" name="cf_<?= h($cf['field_key']) ?>" class="form-check-input"
                         value="1" <?= $cfVal?'checked':'' ?> id="cf_<?= h($cf['field_key']) ?>">
                  <label class="form-check-label" for="cf_<?= h($cf['field_key']) ?>"><?= h($cf['label']) ?></label>
                </div>
              <?php else: ?>
                <input type="<?= $cf['field_type']==='number'?'number':($cf['field_type']==='date'?'date':'text') ?>"
                       name="cf_<?= h($cf['field_key']) ?>" class="form-control"
                       value="<?= h($cfVal) ?>" <?= $cf['is_required']?'required':'' ?>>
              <?php endif; ?>
            </div>
            <?php endforeach; ?>
          </div>
        </div>
      </div>
      <?php endif; ?>

      <!-- Notes -->
      <div class="card mb-4">
        <div class="card-body">
          <p class="form-section-title">Notes</p>
          <textarea name="notes" class="form-control" rows="4" placeholder="Internal notes about this lead…"><?= h($lead['notes']) ?></textarea>
        </div>
      </div>
    </div>

    <!-- Right col -->
    <div class="col-lg-4">
      <div class="card mb-4 position-sticky" style="top:1rem">
        <div class="card-body">
          <p class="form-section-title">Lead Settings</p>
          <div class="mb-3">
            <label class="form-label">Status</label>
            <select name="status" class="form-select">
              <?php foreach(['new'=>'New','in_review'=>'In Review','approved'=>'Approved','declined'=>'Declined','forwarded'=>'Forwarded','on_hold'=>'On Hold'] as $sv=>$sl): ?>
                <option value="<?= $sv ?>" <?= $lead['status']===$sv?'selected':'' ?>><?= $sl ?></option>
              <?php endforeach; ?>
            </select>
          </div>
          <?php if($user['role']!=='agent'): ?>
          <div class="mb-3">
            <label class="form-label">Assign To (Processor)</label>
            <select name="assigned_to" class="form-select">
              <option value="">— Unassigned —</option>
              <?php foreach($processors as $p): ?>
                <option value="<?= $p['id'] ?>" <?= $lead['assigned_to']==$p['id']?'selected':'' ?>>
                  <?= h($p['full_name']?:$p['username']) ?></option>
              <?php endforeach; ?>
            </select>
          </div>
          <?php endif; ?>
          <hr>
          <div class="d-grid gap-2">
            <button type="submit" name="save" class="btn btn-primary">
              <i class="bi bi-check-lg me-1"></i>Save Lead</button>
            <button type="submit" name="save_forward" value="1" class="btn btn-outline-info">
              <i class="bi bi-send me-1"></i>Save & Forward for Processing</button>
            <a href="<?= APP_URL ?>/modules/leads/index.php" class="btn btn-outline-secondary">
              Cancel</a>
          </div>
        </div>
      </div>
    </div>
  </div>
</form>

<?php include __DIR__.'/../../includes/layout_footer.php'; ?>
