<?php
require_once __DIR__.'/../../config/config.php';
require_once __DIR__.'/../../includes/db.php';
require_once __DIR__.'/../../includes/auth.php';
require_once __DIR__.'/../../includes/helpers.php';
session_start_secure();
require_role('admin');

$db   = db();
$user = current_user();

$crmFields = [
    ''                   => '— Skip —',
    'first_name'         => 'First Name',
    'last_name'          => 'Last Name',
    'full_name'          => 'Full Name (auto-split)',
    'email'              => 'Email',
    'phone'              => 'Phone',
    'address'            => 'Address',
    'city'               => 'City',
    'state'              => 'State',
    'zip'                => 'ZIP / Postal Code',
    'dob'                => 'Date of Birth',
    'ssn_last4'          => 'SSN Last 4',
    'annual_income'      => 'Annual Income',
    'employment_status'  => 'Employment Status',
    'credit_score_range' => 'Credit Score Range',
    'requested_limit'    => 'Requested Limit',
    'card_type'          => 'Card Type',
    'card_number_bin'    => 'BIN Number',
    'card_brand'         => 'Card Brand',
    'card_issuer'        => 'Card Issuer / Bank',
    'notes'              => 'Notes',
];

// ── AJAX: poll job ─────────────────────────────────────────
if (isset($_GET['poll_job'])) {
    header('Content-Type: application/json');
    $j = $db->prepare("SELECT * FROM import_jobs WHERE id=? AND user_id=?");
    $j->execute([(int)$_GET['poll_job'], $user['id']]);
    echo json_encode($j->fetch() ?: ['status'=>'error']);
    exit;
}

// ── AJAX: run one chunk ────────────────────────────────────
if (!empty($_POST['run_chunk'])) {
    header('Content-Type: application/json');
    csrf_check();
    $jid  = (int)($_POST['job_id'] ?? 0);
    $CHUNK = 500;

    $job = $db->prepare("SELECT * FROM import_jobs WHERE id=? AND user_id=? AND status IN ('pending','running')");
    $job->execute([$jid, $user['id']]);
    $job = $job->fetch();
    if (!$job) { echo json_encode(['error'=>'Job not found']); exit; }

    $db->prepare("UPDATE import_jobs SET status='running' WHERE id=?")->execute([$jid]);

    $mapping  = json_decode($job['mapping'] ?? '[]', true);
    $delim    = $job['delimiter'];
    $assignTo = $job['assign_to'] ?: $user['id'];
    $fp       = $job['file_path'];

    if (!file_exists($fp)) {
        $db->prepare("UPDATE import_jobs SET status='error',error_log='File not found' WHERE id=?")->execute([$jid]);
        echo json_encode(['error'=>'File missing']); exit;
    }

    $handle   = fopen($fp,'r');
    $skip     = $job['processed'] + 1; // +1 for header
    $lineNum  = 0;
    $imported = 0; $failed = 0; $errors = [];

    while (!feof($handle)) {
        $row = fgetcsv($handle, 0, $delim);
        if ($row === false) continue;
        $lineNum++;
        if ($lineNum <= $skip) continue;
        if ($lineNum > $skip + $CHUNK) break;

        $data = [];
        foreach ($mapping as $idx => $field) {
            if ($field && isset($row[$idx])) $data[$field] = trim($row[$idx]);
        }
        if (!empty($data['full_name']) && empty($data['first_name'])) {
            $p = explode(' ', trim($data['full_name']), 2);
            $data['first_name'] = $p[0] ?? ''; $data['last_name'] = $p[1] ?? '';
            unset($data['full_name']);
        }
        if (empty($data['first_name']) && empty($data['phone']) && empty($data['email'])) {
            $failed++; continue;
        }
        try {
            $db->prepare("INSERT INTO leads
                (ref_number,first_name,last_name,email,phone,dob,ssn_last4,address,city,state,zip,
                 annual_income,employment_status,credit_score_range,requested_limit,card_type,
                 card_number_bin,card_brand,card_issuer,status,agent_id,notes)
                VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,'new',?,?)"
            )->execute([
                generate_ref(),
                $data['first_name'] ?? '', $data['last_name'] ?? '',
                $data['email']      ?? null,
                isset($data['phone']) ? preg_replace('/\D/','',$data['phone']) : null,
                $data['dob']        ?? null,
                isset($data['ssn_last4']) ? preg_replace('/\D/','',$data['ssn_last4']) : null,
                $data['address']    ?? null, $data['city'] ?? null,
                $data['state']      ?? null, $data['zip']  ?? null,
                is_numeric($data['annual_income']  ?? '') ? $data['annual_income']  : null,
                $data['employment_status']          ?? null,
                $data['credit_score_range']         ?? null,
                is_numeric($data['requested_limit'] ?? '') ? $data['requested_limit'] : null,
                $data['card_type']       ?? null,
                isset($data['card_number_bin']) ? preg_replace('/\D/','',$data['card_number_bin']) : null,
                $data['card_brand']      ?? null, $data['card_issuer'] ?? null,
                $assignTo, $data['notes'] ?? null,
            ]);
            $imported++;
        } catch (Exception $e) {
            $failed++;
            if (count($errors)<20) $errors[] = "Row $lineNum: ".$e->getMessage();
        }
    }
    fclose($handle);

    $newProc = $job['processed'] + $imported + $failed;
    $newImp  = $job['imported']  + $imported;
    $newFail = $job['failed']    + $failed;
    $done    = ($newProc >= $job['total_rows']);
    $st      = $done ? 'done' : 'running';
    $errLog  = trim(($job['error_log']??'')."\n".implode("\n",$errors));
    if ($done) @unlink($fp);

    $db->prepare("UPDATE import_jobs SET processed=?,imported=?,failed=?,status=?,error_log=?,
        finished_at=".($done?'NOW()':'NULL')." WHERE id=?")
       ->execute([$newProc,$newImp,$newFail,$st,$errLog,$jid]);

    if ($done) audit('bulk_import_done','lead',null,"Job $jid: $newImp imported, $newFail failed");

    echo json_encode(['status'=>$st,'processed'=>$newProc,'imported'=>$newImp,'failed'=>$newFail,
        'total'=>$job['total_rows'],'job_id'=>$jid,'error_log'=>$done?$errLog:null]);
    exit;
}

// ── Normal page ────────────────────────────────────────────
$step = 'upload';
$headers = []; $previewRows = []; $detectedDelim = ',';
$errors = []; $jobId = 0;
$agents = $db->query("SELECT id,full_name,username FROM users WHERE role IN ('agent','admin') AND is_active=1 ORDER BY full_name")->fetchAll();
$recentJobs = $db->query("SELECT ij.*,u.full_name as creator FROM import_jobs ij LEFT JOIN users u ON u.id=ij.user_id ORDER BY ij.created_at DESC LIMIT 8")->fetchAll();

if ($_SERVER['REQUEST_METHOD']==='POST') {
    csrf_check();
    $postStep = $_POST['step'] ?? 'upload';

    if ($postStep==='upload') {
        $rawText=''; $filePath=''; $origName='';
        if (!empty($_FILES['upload_file']['tmp_name'])) {
            $file=$_FILES['upload_file'];
            $ext=strtolower(pathinfo($file['name'],PATHINFO_EXTENSION));
            if ($file['size']>200*1024*1024) $errors[]='Max 200MB.';
            elseif (!in_array($ext,['csv','txt','tsv','dat'])) $errors[]='Supported: .csv .txt .tsv .dat';
            else {
                $dir=sys_get_temp_dir().'/crm_imports/'; @mkdir($dir,0700,true);
                $dest=$dir.'import_'.uniqid().'.'.$ext;
                move_uploaded_file($file['tmp_name'],$dest);
                $filePath=$dest; $origName=$file['name'];
            }
        } elseif (($rawText=trim($_POST['paste_text']??''))!=='') {
            $dir=sys_get_temp_dir().'/crm_imports/'; @mkdir($dir,0700,true);
            $filePath=$dir.'paste_'.uniqid().'.txt';
            file_put_contents($filePath,$rawText);
            $origName='pasted_data.txt';
        } else { $errors[]='Upload a file or paste data.'; }

        if (!$errors && $filePath) {
            // Detect delimiter
            $h=fopen($filePath,'r'); $sample='';
            for($i=0;$i<5&&!feof($h);$i++) $sample.=fgets($h); fclose($h);
            $counts=[','  =>substr_count($sample,','),"\t"=>substr_count($sample,"\t"),
                     '|'  =>substr_count($sample,'|'), ';' =>substr_count($sample,';')];
            arsort($counts); $detectedDelim=array_key_first($counts);
            $totalRows=max(0,count(file($filePath,FILE_SKIP_EMPTY_LINES))-1);
            $h=fopen($filePath,'r');
            $headers=fgetcsv($h,0,$detectedDelim)?:[];
            $previewRows=[];
            for($i=0;$i<6;$i++){$r=fgetcsv($h,0,$detectedDelim);if($r===false)break;$previewRows[]=$r;}
            fclose($h);
            $_SESSION['imp_file']=$filePath; $_SESSION['imp_name']=$origName;
            $_SESSION['imp_delim']=$detectedDelim; $_SESSION['imp_total']=$totalRows;
            $step='map';
        }
    }

    if ($postStep==='map') {
        $filePath=$_SESSION['imp_file']??''; $origName=$_SESSION['imp_name']??'';
        $delim=$_POST['delimiter']??($_SESSION['imp_delim']??',');
        $mapping=$_POST['map']??[]; $assignTo=(int)($_POST['assign_to']??$user['id']);
        if (!$filePath||!file_exists($filePath)){$errors[]='Session expired.';$step='upload';}
        elseif (!array_filter($mapping)){$errors[]='Map at least one column.';$step='map';}
        else {
            $totalRows=max(0,count(file($filePath,FILE_SKIP_EMPTY_LINES))-1);
            $db->prepare("INSERT INTO import_jobs (user_id,filename,file_path,delimiter,mapping,assign_to,total_rows,status) VALUES (?,?,?,?,?,?,?,'pending')")
               ->execute([$user['id'],$origName,$filePath,$delim,json_encode($mapping),$assignTo,$totalRows]);
            $jobId=(int)$db->lastInsertId();
            unset($_SESSION['imp_file'],$_SESSION['imp_name'],$_SESSION['imp_delim'],$_SESSION['imp_total']);
            audit('bulk_import_start','lead',null,"Job $jobId: $totalRows rows");
            $step='running';
        }
    }
}

if ($step==='map'&&empty($headers)&&!empty($_SESSION['imp_file'])) {
    $fp=$_SESSION['imp_file']; $dl=$_SESSION['imp_delim']??',';
    if (file_exists($fp)) {
        $h=fopen($fp,'r'); $headers=fgetcsv($h,0,$dl)?:[];
        $previewRows=[]; for($i=0;$i<5;$i++){$r=fgetcsv($h,0,$dl);if($r)$previewRows[]=$r;} fclose($h);
        $detectedDelim=$dl;
    }
}

$pageTitle='Bulk Import';
include __DIR__.'/../../includes/layout_header.php';
?>

<div class="page-header d-flex align-items-start justify-content-between flex-wrap gap-2">
  <div>
    <h1><i class="bi bi-cloud-upload me-2 text-primary fs-4"></i>Bulk Import Leads</h1>
    <p class="text-muted small mb-0">Upload or paste delimited data — CSV, TXT, TSV — 200MB max, 100k+ rows</p>
  </div>
</div>

<!-- Steps -->
<div class="d-flex align-items-center gap-2 mb-4 small">
  <?php foreach(['upload'=>'1 Upload / Paste','map'=>'2 Map Columns','running'=>'3 Importing'] as $k=>$l): ?>
    <span class="badge rounded-pill <?= $step===$k?'bg-primary':($step==='running'&&$k!=='running'?'bg-success':'bg-secondary') ?>">
      <?php if($step==='running'&&$k!=='running'): ?><i class="bi bi-check me-1"></i><?php endif; ?>
      <?= $l ?></span>
    <?php if($k!=='running'): ?><i class="bi bi-chevron-right text-muted"></i><?php endif; ?>
  <?php endforeach; ?>
</div>

<?php if($errors): ?>
<div class="alert alert-danger mb-3"><i class="bi bi-exclamation-triangle me-1"></i><?= implode('<br>',$errors) ?></div>
<?php endif; ?>

<?php if($step==='upload'): ?>
<div class="row g-4">
  <div class="col-lg-6">
    <div class="card h-100">
      <div class="card-header fw-semibold"><i class="bi bi-file-earmark-arrow-up me-2 text-primary"></i>Upload File</div>
      <div class="card-body">
        <form method="post" enctype="multipart/form-data">
          <input type="hidden" name="csrf_token" value="<?= csrf_token() ?>">
          <input type="hidden" name="step" value="upload">
          <div class="drop-zone mb-3" id="dropZone">
            <i class="bi bi-file-earmark-spreadsheet d-block mb-2 fs-2 opacity-40"></i>
            <div class="fw-medium mb-1">Drag & drop file here</div>
            <div class="text-muted small mb-3">CSV · TXT · TSV · DAT &nbsp;|&nbsp; Up to 200MB &nbsp;|&nbsp; 100k+ rows</div>
            <input type="file" name="upload_file" id="fileInput" accept=".csv,.txt,.tsv,.dat" class="d-none">
            <button type="button" class="btn btn-outline-primary btn-sm"
              onclick="document.getElementById('fileInput').click()">
              <i class="bi bi-folder2-open me-1"></i>Choose File</button>
            <div id="fileNameDisplay" class="mt-2 small fw-semibold text-primary"></div>
          </div>
          <div class="alert alert-info small py-2 mb-3">
            <i class="bi bi-info-circle me-1"></i>
            First row must be a <strong>header row</strong>. Delimiter (comma, tab, pipe, semicolon)
            is auto-detected. The import runs in background chunks — you can watch it live.
          </div>
          <button type="submit" class="btn btn-primary w-100" id="uploadBtn" disabled>
            <i class="bi bi-arrow-right me-1"></i>Continue to Column Mapping</button>
        </form>
      </div>
    </div>
  </div>
  <div class="col-lg-6">
    <div class="card h-100">
      <div class="card-header fw-semibold"><i class="bi bi-clipboard me-2 text-primary"></i>Paste Data</div>
      <div class="card-body d-flex flex-column">
        <p class="text-muted small mb-2">
          Paste any delimited text — comma, tab, pipe, or semicolon separated.
          <strong>Include a header row</strong> as the first line.
        </p>
        <p class="text-muted small mb-2 fst-italic">Example:<br>
          <code style="font-size:.75rem">first_name,last_name,phone,email<br>John,Smith,5551234567,john@example.com</code>
        </p>
        <form method="post" class="flex-grow-1 d-flex flex-column">
          <input type="hidden" name="csrf_token" value="<?= csrf_token() ?>">
          <input type="hidden" name="step" value="upload">
          <textarea name="paste_text" id="pasteText"
            class="form-control mono flex-grow-1 mb-3"
            style="min-height:190px;font-size:.78rem;resize:vertical"
            placeholder="first_name,last_name,phone,email&#10;John,Smith,5551234567,john@example.com&#10;Jane,Doe,5559876543,jane@example.com"></textarea>
          <button type="submit" class="btn btn-outline-primary w-100" id="pasteBtn" disabled>
            <i class="bi bi-arrow-right me-1"></i>Continue to Column Mapping</button>
        </form>
      </div>
    </div>
  </div>
</div>

<?php if($recentJobs): ?>
<div class="mt-4">
  <h6 class="text-muted small text-uppercase mb-2">Recent Import History</h6>
  <div class="card">
    <div class="table-responsive">
      <table class="table table-sm mb-0">
        <thead><tr><th>File</th><th>Total</th><th>Imported</th><th>Failed</th><th>Status</th><th>By</th><th>Date</th></tr></thead>
        <tbody>
          <?php foreach($recentJobs as $j): ?>
          <tr>
            <td class="small"><?= h(basename($j['filename'])) ?></td>
            <td class="small"><?= number_format($j['total_rows']) ?></td>
            <td class="small text-success fw-medium"><?= number_format($j['imported']) ?></td>
            <td class="small <?= $j['failed']>0?'text-danger':'' ?>"><?= number_format($j['failed']) ?></td>
            <td><?php $sc=['pending'=>'bg-secondary','running'=>'bg-primary','done'=>'bg-success','error'=>'bg-danger'];
              echo '<span class="badge '.($sc[$j['status']]??'bg-secondary').'">'.h($j['status']).'</span>'; ?></td>
            <td class="small text-muted"><?= h($j['creator']??'—') ?></td>
            <td class="small text-muted"><?= h(date('M j, Y g:ia',strtotime($j['created_at']))) ?></td>
          </tr>
          <?php endforeach; ?>
        </tbody>
      </table>
    </div>
  </div>
</div>
<?php endif; ?>

<?php elseif($step==='map'): ?>
<div class="card mb-4">
  <div class="card-header d-flex align-items-center justify-content-between">
    <span class="fw-semibold"><i class="bi bi-diagram-3 me-2 text-primary"></i>Map File Columns to CRM Fields</span>
    <span class="badge bg-secondary"><?= number_format($_SESSION['imp_total']??0) ?> rows to import</span>
  </div>
  <div class="card-body">
    <form method="post">
      <input type="hidden" name="csrf_token" value="<?= csrf_token() ?>">
      <input type="hidden" name="step" value="map">

      <div class="row g-3 mb-4 align-items-end">
        <div class="col-md-3">
          <label class="form-label">Column Delimiter <span class="badge bg-success ms-1">Auto-detected</span></label>
          <select name="delimiter" class="form-select form-select-sm">
            <option value="," <?= $detectedDelim===','?'selected':'' ?>>Comma ( , ) — CSV</option>
            <option value="	" <?= $detectedDelim==="\t"?'selected':'' ?>>Tab — TSV</option>
            <option value="|" <?= $detectedDelim==='|'?'selected':'' ?>>Pipe ( | )</option>
            <option value=";" <?= $detectedDelim===';'?'selected':'' ?>>Semicolon ( ; )</option>
          </select>
        </div>
        <div class="col-md-4">
          <label class="form-label">Assign Imported Leads To</label>
          <select name="assign_to" class="form-select form-select-sm">
            <option value="<?= $user['id'] ?>">Me (<?= h($user['full_name']?:$user['username']) ?>)</option>
            <?php foreach($agents as $a): if($a['id']==$user['id']) continue; ?>
              <option value="<?= $a['id'] ?>"><?= h($a['full_name']?:$a['username']) ?></option>
            <?php endforeach; ?>
          </select>
        </div>
        <div class="col text-end">
          <a href="<?= APP_URL ?>/modules/leads/bulk_upload.php" class="btn btn-outline-secondary btn-sm">
            <i class="bi bi-arrow-left me-1"></i>Start Over</a>
        </div>
      </div>

      <div class="table-responsive mb-4">
        <table class="table table-sm table-bordered align-middle">
          <thead class="table-light">
            <tr><th style="width:36px">#</th><th>File Column</th><th>Sample Data</th><th style="min-width:200px">Map to CRM Field</th></tr>
          </thead>
          <tbody>
            <?php foreach($headers as $i=>$hdr):
              // Smart auto-guess
              $colKey=strtolower(preg_replace('/[^a-z0-9]/i','_',$hdr));
              $aliases=['fname'=>'first_name','lname'=>'last_name','lastname'=>'last_name','firstname'=>'first_name',
                        'mobile'=>'phone','cell'=>'phone','telephone'=>'phone','tel'=>'phone','name'=>'full_name',
                        'fullname'=>'full_name','full_name'=>'full_name','income'=>'annual_income',
                        'bank'=>'card_issuer','bin'=>'card_number_bin','zipcode'=>'zip','postal'=>'zip',
                        'postalcode'=>'zip','postcode'=>'zip','addr'=>'address','street'=>'address',
                        'dob'=>'dob','birthdate'=>'dob','birthday'=>'dob','ssn'=>'ssn_last4',
                        'last4'=>'ssn_last4','employer'=>'employment_status','employment'=>'employment_status',
                        'credit'=>'credit_score_range','score'=>'credit_score_range','limit'=>'requested_limit',
                        'issuer'=>'card_issuer','brand'=>'card_brand','cardtype'=>'card_type'];
              $best=$aliases[$colKey]??'';
              if(!$best){
                $bScore=0;
                foreach(array_filter(array_keys($crmFields)) as $cf){
                    similar_text($colKey,$cf,$p);
                    if($p>$bScore&&$p>62){$bScore=$p;$best=$cf;}
                }
              }
              $samples=[];
              foreach(array_slice($previewRows,0,3) as $pr){
                  if(isset($pr[$i])&&trim($pr[$i])!=='') $samples[]=$pr[$i];
              }
            ?>
            <tr>
              <td class="text-muted small"><?= $i+1 ?></td>
              <td class="fw-medium small"><?= h($hdr) ?></td>
              <td class="text-muted" style="font-size:.75rem;max-width:180px"><?= h(implode(' / ',array_slice($samples,0,3))) ?></td>
              <td>
                <select name="map[<?= $i ?>]" class="form-select form-select-sm">
                  <?php foreach($crmFields as $val=>$lbl): ?>
                    <option value="<?= h($val) ?>" <?= $best===$val?'selected':'' ?>><?= h($lbl) ?></option>
                  <?php endforeach; ?>
                </select>
              </td>
            </tr>
            <?php endforeach; ?>
          </tbody>
        </table>
      </div>

      <!-- Preview -->
      <h6 class="text-muted small text-uppercase mb-2"><i class="bi bi-eye me-1"></i>Data Preview (first <?= count($previewRows) ?> rows)</h6>
      <div class="table-responsive mb-4" style="max-height:220px;overflow:auto">
        <table class="table table-sm table-bordered table-striped" style="font-size:.73rem;white-space:nowrap">
          <thead class="sticky-top table-light">
            <tr><?php foreach($headers as $hh): ?><th><?= h($hh) ?></th><?php endforeach; ?></tr>
          </thead>
          <tbody>
            <?php foreach($previewRows as $pr): ?>
            <tr><?php foreach($pr as $cell): ?><td><?= h($cell) ?></td><?php endforeach; ?></tr>
            <?php endforeach; ?>
          </tbody>
        </table>
      </div>

      <button type="submit" class="btn btn-primary">
        <i class="bi bi-lightning-charge me-1"></i>
        Start Importing <?= number_format($_SESSION['imp_total']??0) ?> rows</button>
    </form>
  </div>
</div>

<?php elseif($step==='running'): ?>
<div class="card mx-auto" style="max-width:600px">
  <div class="card-body py-5">
    <div class="text-center mb-4">
      <div id="importIcon"><div class="spinner-border text-primary mb-2" style="width:3rem;height:3rem"></div></div>
      <h5 class="fw-bold mb-1" id="importTitle">Importing Leads…</h5>
      <p class="text-muted small" id="importSub">Starting…</p>
    </div>
    <div class="mb-4">
      <div class="d-flex justify-content-between small mb-1">
        <span id="progLabel">Initializing…</span><span id="progPct">0%</span>
      </div>
      <div class="progress" style="height:14px;border-radius:10px">
        <div id="progBar" class="progress-bar progress-bar-striped progress-bar-animated"
             style="width:1%;border-radius:10px;transition:width .2s ease"></div>
      </div>
    </div>
    <div class="row g-3 text-center mb-4">
      <div class="col-4">
        <div class="fw-bold fs-3 text-success" id="cntImported">0</div>
        <div class="text-muted small">Imported</div>
      </div>
      <div class="col-4">
        <div class="fw-bold fs-3 text-danger" id="cntFailed">0</div>
        <div class="text-muted small">Failed</div>
      </div>
      <div class="col-4">
        <div class="fw-bold fs-3 text-muted" id="cntRemaining">—</div>
        <div class="text-muted small">Remaining</div>
      </div>
    </div>
    <div id="doneSection" style="display:none">
      <div class="alert alert-success text-center py-2 fw-medium" id="doneMsg"></div>
      <div id="errSection" style="display:none">
        <details><summary class="small text-danger fw-medium">View import errors</summary>
          <pre id="errLog" class="small mt-2 p-2 rounded" style="max-height:140px;overflow:auto;font-size:.7rem;background:var(--bs-tertiary-bg)"></pre>
        </details>
      </div>
      <div class="d-flex gap-2 justify-content-center mt-3">
        <a href="<?= APP_URL ?>/modules/leads/index.php" class="btn btn-primary">View Leads</a>
        <a href="<?= APP_URL ?>/modules/leads/bulk_upload.php" class="btn btn-outline-secondary">Import More</a>
      </div>
    </div>
    <div id="runningControls">
      <button class="btn btn-outline-secondary btn-sm w-100" onclick="if(confirm('Cancel?'))cancelled=true">
        <i class="bi bi-stop-circle me-1"></i>Cancel</button>
    </div>
  </div>
</div>

<script>
let cancelled=false, totalRows=0;
const JOB=<?= $jobId ?>, CSRF='<?= csrf_token() ?>';
const URL='<?= APP_URL ?>/modules/leads/bulk_upload.php';

async function chunk(){
  if(cancelled)return;
  try{
    const r=await fetch(URL,{method:'POST',headers:{'Content-Type':'application/x-www-form-urlencoded'},
      body:'run_chunk=1&job_id='+JOB+'&csrf_token='+encodeURIComponent(CSRF)});
    const d=await r.json();
    if(d.error){setTitle('Error',d.error,'text-danger');return;}
    totalRows=d.total||totalRows;
    const pct=totalRows>0?Math.min(100,Math.round(d.processed/totalRows*100)):0;
    document.getElementById('progBar').style.width=Math.max(2,pct)+'%';
    document.getElementById('progLabel').textContent=d.processed.toLocaleString()+' / '+totalRows.toLocaleString()+' rows';
    document.getElementById('progPct').textContent=pct+'%';
    document.getElementById('cntImported').textContent=d.imported.toLocaleString();
    document.getElementById('cntFailed').textContent=d.failed.toLocaleString();
    document.getElementById('cntRemaining').textContent=Math.max(0,totalRows-d.processed).toLocaleString();
    document.getElementById('importSub').textContent='Row '+d.processed.toLocaleString()+' processed…';
    if(d.status==='done') done(d);
    else setTimeout(chunk,30);
  }catch(e){setTitle('Network Error',e.message,'text-danger');}
}

function done(d){
  document.getElementById('progBar').className='progress-bar bg-success';
  document.getElementById('progBar').style.width='100%';
  document.getElementById('importIcon').innerHTML='<i class="bi bi-check-circle-fill text-success fs-1 d-block mb-2"></i>';
  document.getElementById('importTitle').textContent='Import Complete!';
  document.getElementById('importSub').textContent='';
  document.getElementById('runningControls').style.display='none';
  document.getElementById('doneSection').style.display='';
  document.getElementById('doneMsg').textContent=d.imported.toLocaleString()+' leads imported'+(d.failed>0?' — '+d.failed+' rows failed.':'.');
  if(d.failed>0&&d.error_log){document.getElementById('errSection').style.display='';document.getElementById('errLog').textContent=d.error_log;}
}
function setTitle(t,s,cls){
  document.getElementById('importTitle').textContent=t;
  document.getElementById('importSub').textContent=s;
  document.getElementById('progBar').className='progress-bar '+cls;
}
setTimeout(chunk,400);
</script>
<?php endif; ?>

<script>
const fi=document.getElementById('fileInput');
const fn=document.getElementById('fileNameDisplay');
const ub=document.getElementById('uploadBtn');
const dz=document.getElementById('dropZone');
const pt=document.getElementById('pasteText');
const pb=document.getElementById('pasteBtn');
if(fi){fi.addEventListener('change',()=>{fn.textContent=fi.files[0]?.name||'';if(ub)ub.disabled=!fi.files[0];});}
if(dz){
  dz.addEventListener('dragover',e=>{e.preventDefault();dz.classList.add('dragover');});
  dz.addEventListener('dragleave',()=>dz.classList.remove('dragover'));
  dz.addEventListener('drop',e=>{e.preventDefault();dz.classList.remove('dragover');
    const f=e.dataTransfer.files[0];if(!f)return;
    const dt=new DataTransfer();dt.items.add(f);fi.files=dt.files;
    fn.textContent=f.name;if(ub)ub.disabled=false;});
  dz.addEventListener('click',e=>{if(!e.target.closest('button'))fi?.click();});
}
if(pt&&pb){pt.addEventListener('input',()=>{pb.disabled=pt.value.trim().length<10;});}
</script>

<?php include __DIR__.'/../../includes/layout_footer.php'; ?>
