<?php
require_once __DIR__.'/../../config/config.php';
require_once __DIR__.'/../../includes/db.php';
require_once __DIR__.'/../../includes/auth.php';
require_once __DIR__.'/../../includes/helpers.php';
session_start_secure();
require_login();

$user = current_user();
$db   = db();

// ── Handle quick call outcome log (AJAX) ────────────────
if (!empty($_POST['log_call'])) {
    header('Content-Type: application/json');
    csrf_check();
    $lid     = (int)($_POST['lead_id'] ?? 0);
    $outcome = $_POST['outcome'] ?? '';
    $notes   = trim($_POST['notes'] ?? '');
    $allowed = ['no_answer','voicemail','callback','picked_up','wrong_number','do_not_call'];
    if (!$lid || !in_array($outcome, $allowed)) {
        echo json_encode(['ok'=>false,'error'=>'Bad input']); exit;
    }
    // Access check
    if ($user['role']==='agent') {
        $chk = $db->prepare("SELECT id FROM leads WHERE id=? AND agent_id=?");
        $chk->execute([$lid,$user['id']]);
        if (!$chk->fetch()) { echo json_encode(['ok'=>false,'error'=>'Access denied']); exit; }
    }
    $db->prepare("INSERT INTO call_log (lead_id,agent_id,outcome,notes) VALUES (?,?,?,?)")
       ->execute([$lid,$user['id'],$outcome,$notes]);

    // Auto status updates
    $statusMap = [
        'picked_up'      => 'in_review',
        'do_not_call'    => 'declined',
        'callback'       => 'on_hold',
    ];
    if (isset($statusMap[$outcome])) {
        $db->prepare("UPDATE leads SET status=? WHERE id=?")->execute([$statusMap[$outcome],$lid]);
    }
    audit('call_logged','lead',$lid,"Outcome: $outcome".($notes?" | $notes":''));
    echo json_encode(['ok'=>true,'outcome'=>$outcome,'new_status'=>$statusMap[$outcome]??null]);
    exit;
}

// ── Filters & queue build ───────────────────────────────
$statusFilter = $_GET['status'] ?? 'new';
$search       = trim($_GET['q'] ?? '');
$sort         = in_array($_GET['sort']??'',['created_at','last_name','phone']) ? $_GET['sort'] : 'created_at';
$dir          = ($_GET['dir']??'asc')==='desc' ? 'DESC' : 'ASC';
$page         = max(1,(int)($_GET['page']??1));
$perPage      = 25;

$where  = ['1'];
$params = [];

// Agents only see own leads; admins/processors see all
if ($user['role']==='agent') { $where[]='l.agent_id=?'; $params[]=$user['id']; }

if ($statusFilter && $statusFilter!=='all') {
    $where[]='l.status=?'; $params[]=$statusFilter;
}
if ($search) {
    $s='%'.$search.'%';
    $where[]='(l.first_name LIKE ? OR l.last_name LIKE ? OR l.phone LIKE ? OR l.email LIKE ? OR l.ref_number LIKE ?)';
    array_push($params,$s,$s,$s,$s,$s);
}
$whereStr = implode(' AND ',$where);

$total = (int)$db->prepare("SELECT COUNT(*) FROM leads l WHERE $whereStr")->execute($params) ? 0 : 0;
$cSt   = $db->prepare("SELECT COUNT(*) FROM leads l WHERE $whereStr"); $cSt->execute($params);
$total = (int)$cSt->fetchColumn();
$pages = max(1,(int)ceil($total/$perPage));
$page  = min($page,$pages);
$off   = ($page-1)*$perPage;

$leads = $db->prepare("SELECT l.*, COALESCE(u.full_name,u.username) AS agent_name,
    (SELECT COUNT(*) FROM call_log cl WHERE cl.lead_id=l.id) AS call_count,
    (SELECT cl2.outcome FROM call_log cl2 WHERE cl2.lead_id=l.id ORDER BY cl2.created_at DESC LIMIT 1) AS last_outcome,
    (SELECT cl3.created_at FROM call_log cl3 WHERE cl3.lead_id=l.id ORDER BY cl3.created_at DESC LIMIT 1) AS last_called
    FROM leads l LEFT JOIN users u ON u.id=l.agent_id
    WHERE $whereStr ORDER BY l.$sort $dir LIMIT $perPage OFFSET $off");
$leads->execute($params);
$leads = $leads->fetchAll();

// Counts by status for tabs
$counts = [];
foreach (['new','in_review','on_hold','forwarded','all'] as $s) {
    $w2 = $user['role']==='agent' ? "agent_id={$user['id']}" : '1';
    $addW = $s==='all' ? '' : " AND status='$s'";
    $counts[$s] = (int)$db->query("SELECT COUNT(*) FROM leads WHERE $w2$addW")->fetchColumn();
}

$pageTitle = 'Call Queue';
include __DIR__.'/../../includes/layout_header.php';
?>

<div class="page-header d-flex align-items-center justify-content-between flex-wrap gap-2">
  <div>
    <h1><i class="bi bi-telephone-outbound me-2 text-primary fs-4"></i>Call Queue</h1>
    <p class="text-muted small mb-0">Work through your leads, log call outcomes, and forward eligible customers to processing</p>
  </div>
</div>

<!-- Status tabs -->
<div class="mb-3">
  <ul class="nav nav-tabs">
    <?php
    $tabs = [
        'new'       => ['New',       'badge-new'],
        'in_review' => ['In Review', 'badge-review'],
        'on_hold'   => ['Callback',  'badge-hold'],
        'all'       => ['All',       'bg-secondary'],
    ];
    foreach ($tabs as $sv => [$label, $badgeCls]):
        $isActive = $statusFilter === $sv;
        $href     = '?status='.$sv.($search?'&q='.urlencode($search):'');
    ?>
    <li class="nav-item">
      <a class="nav-link py-2 <?= $isActive?'active':'' ?>" href="<?= $href ?>">
        <?= $label ?>
        <span class="ms-1 badge rounded-pill <?= $badgeCls ?>" style="font-size:.65rem">
          <?= number_format($counts[$sv]??0) ?></span>
      </a>
    </li>
    <?php endforeach; ?>
  </ul>
</div>

<!-- Search -->
<form method="get" class="mb-3 d-flex gap-2" style="max-width:420px">
  <input type="hidden" name="status" value="<?= h($statusFilter) ?>">
  <div class="input-group input-group-sm">
    <span class="input-group-text"><i class="bi bi-search"></i></span>
    <input type="text" name="q" class="form-control" placeholder="Search name, phone, email…" value="<?= h($search) ?>">
    <?php if($search): ?><a href="?status=<?= h($statusFilter) ?>" class="btn btn-outline-secondary">×</a><?php endif; ?>
  </div>
  <button type="submit" class="btn btn-sm btn-primary">Search</button>
</form>

<!-- Lead cards grid -->
<?php if ($leads): ?>
<div class="row g-3 mb-4" id="queueGrid">
  <?php foreach ($leads as $lead): ?>
  <?php
    $callCount   = (int)$lead['call_count'];
    $lastOutcome = $lead['last_outcome'] ?? '';
    $outcomeIcons = [
        'no_answer'    => ['bi-telephone-x text-secondary','No Answer'],
        'voicemail'    => ['bi-voicemail text-secondary','Voicemail'],
        'callback'     => ['bi-arrow-clockwise text-warning','Callback Requested'],
        'picked_up'    => ['bi-telephone-fill text-success','Picked Up'],
        'wrong_number' => ['bi-telephone-minus text-danger','Wrong Number'],
        'do_not_call'  => ['bi-ban text-danger','Do Not Call'],
    ];
    [$oIcon,$oLabel] = $outcomeIcons[$lastOutcome] ?? ['bi-dash text-muted',''];
  ?>
  <div class="col-md-6 col-xl-4 queue-card-wrap" id="card-<?= $lead['id'] ?>">
    <div class="card h-100 queue-card">
      <div class="card-body pb-2">
        <!-- Name + status -->
        <div class="d-flex align-items-start justify-content-between gap-2 mb-2">
          <div>
            <div class="fw-semibold">
              <a href="<?= APP_URL ?>/modules/leads/view.php?id=<?= $lead['id'] ?>" target="_blank"
                 class="text-decoration-none">
                <?= h($lead['first_name'].' '.$lead['last_name']) ?>
                <i class="bi bi-box-arrow-up-right ms-1 text-muted" style="font-size:.65rem"></i>
              </a>
            </div>
            <div class="mono text-muted" style="font-size:.65rem"><?= h($lead['ref_number']) ?></div>
          </div>
          <?= status_badge($lead['status']) ?>
        </div>

        <!-- Phone — big click-to-call button -->
        <div class="mb-2">
          <?php if ($lead['phone']): ?>
          <a href="tel:<?= h(preg_replace('/\D/','',$lead['phone'])) ?>"
             class="btn btn-success btn-sm w-100 d-flex align-items-center justify-content-center gap-2"
             style="font-size:1rem;padding:.5rem">
            <i class="bi bi-telephone-fill"></i>
            <span class="fw-bold mono"><?= h(format_phone($lead['phone'])) ?></span>
          </a>
          <?php else: ?>
          <div class="btn btn-secondary btn-sm w-100 disabled">No phone on file</div>
          <?php endif; ?>
        </div>

        <!-- Quick info row -->
        <div class="d-flex flex-wrap gap-2 mb-2 small text-muted">
          <?php if ($lead['city']||$lead['state']): ?>
          <span><i class="bi bi-geo-alt me-1"></i><?= h(trim($lead['city'].', '.$lead['state'],', ')) ?></span>
          <?php endif; ?>
          <?php if ($lead['card_issuer']): ?>
          <span><i class="bi bi-bank me-1"></i><?= h($lead['card_issuer']) ?></span>
          <?php endif; ?>
        </div>

        <!-- Call history mini -->
        <?php if ($callCount > 0): ?>
        <div class="mb-2 small">
          <i class="bi <?= $oIcon ?> me-1"></i>
          <span class="text-muted">Last: <?= $oLabel ?></span>
          <span class="text-muted ms-2"><i class="bi bi-telephone me-1"></i><?= $callCount ?> call<?= $callCount>1?'s':'' ?></span>
          <?php if($lead['last_called']): ?>
            <span class="text-muted ms-1">· <?= h(date('M j',strtotime($lead['last_called']))) ?></span>
          <?php endif; ?>
        </div>
        <?php endif; ?>
      </div>

      <!-- Action buttons -->
      <div class="card-footer py-2 px-3 d-flex gap-1 flex-wrap">
        <button class="btn btn-outline-success btn-sm flex-grow-1"
          onclick="openCallModal(<?= $lead['id'] ?>, '<?= h(addslashes($lead['first_name'].' '.$lead['last_name'])) ?>')"
          title="Log call outcome">
          <i class="bi bi-journal-check me-1"></i>Log Call</button>

        <a href="<?= APP_URL ?>/modules/leads/edit.php?id=<?= $lead['id'] ?>"
           class="btn btn-outline-primary btn-sm" title="Edit lead details">
          <i class="bi bi-pencil"></i></a>

        <?php if ($lead['status'] !== 'forwarded'): ?>
        <button class="btn btn-outline-info btn-sm"
          onclick="forwardLead(<?= $lead['id'] ?>, '<?= h(addslashes($lead['first_name'].' '.$lead['last_name'])) ?>')"
          title="Forward to processor — customer is eligible">
          <i class="bi bi-send"></i></button>
        <?php else: ?>
        <button class="btn btn-info btn-sm disabled" title="Already forwarded"><i class="bi bi-check2"></i></button>
        <?php endif; ?>
      </div>
    </div>
  </div>
  <?php endforeach; ?>
</div>

<!-- Pagination -->
<?php if ($pages > 1): ?>
<nav class="mb-4"><ul class="pagination pagination-sm">
  <?php $qp=$_GET; unset($qp['page']); $base='?'.http_build_query($qp).'&page=';
  if($page>1): ?><li class="page-item"><a class="page-link" href="<?= $base.($page-1) ?>">‹</a></li><?php endif; ?>
  <?php for($i=max(1,$page-2);$i<=min($pages,$page+2);$i++): ?>
    <li class="page-item <?= $i===$page?'active':'' ?>"><a class="page-link" href="<?= $base.$i ?>"><?= $i ?></a></li>
  <?php endfor; ?>
  <?php if($page<$pages): ?><li class="page-item"><a class="page-link" href="<?= $base.($page+1) ?>">›</a></li><?php endif; ?>
</ul></nav>
<?php endif; ?>

<?php else: ?>
<div class="card"><div class="card-body text-center py-5 text-muted">
  <i class="bi bi-inbox fs-2 d-block mb-2 opacity-25"></i>
  <div class="fw-medium">No leads in this queue</div>
  <p class="small mt-1 mb-0">
    <?php if($statusFilter==='new'): ?>
      New leads will appear here after a bulk import or when added manually.
    <?php else: ?>
      No leads match your current filter.
    <?php endif; ?>
  </p>
</div></div>
<?php endif; ?>

<!-- ── Call Log Modal ────────────────────────────────────── -->
<div class="modal fade" id="callModal" tabindex="-1">
  <div class="modal-dialog modal-sm">
    <div class="modal-content">
      <div class="modal-header py-2">
        <h6 class="modal-title mb-0"><i class="bi bi-telephone me-2 text-primary"></i>Log Call: <span id="callModalName"></span></h6>
        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
      </div>
      <div class="modal-body">
        <p class="small text-muted mb-3">What was the outcome of this call?</p>
        <div class="d-grid gap-2" id="outcomeButtons">
          <!-- filled by JS -->
        </div>
        <div class="mt-3">
          <label class="form-label small mb-1">Quick note (optional)</label>
          <textarea id="callNotes" class="form-control form-control-sm" rows="2" placeholder="e.g. left voicemail, call back after 5pm…"></textarea>
        </div>
      </div>
    </div>
  </div>
</div>

<!-- ── Forward confirm modal ─────────────────────────────── -->
<div class="modal fade" id="forwardModal" tabindex="-1">
  <div class="modal-dialog modal-sm">
    <div class="modal-content">
      <div class="modal-header py-2">
        <h6 class="modal-title mb-0"><i class="bi bi-send me-2 text-info"></i>Forward to Processing</h6>
        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
      </div>
      <div class="modal-body">
        <p class="mb-2">Forward <strong id="forwardName"></strong> to the processing queue?</p>
        <p class="small text-muted mb-0">The lead will be marked as <strong>Forwarded</strong> and will appear in the processor's queue. Make sure all required information has been filled in first.</p>
      </div>
      <div class="modal-footer py-2">
        <button class="btn btn-secondary btn-sm" data-bs-dismiss="modal">Cancel</button>
        <button class="btn btn-info btn-sm" id="forwardConfirmBtn">
          <i class="bi bi-send me-1"></i>Yes, Forward It</button>
      </div>
    </div>
  </div>
</div>

<style>
.queue-card { transition: box-shadow .15s; }
.queue-card:hover { box-shadow: 0 4px 16px rgba(0,0,0,.12); }
.queue-card.dimmed { opacity: .45; pointer-events: none; }
.outcome-btn { text-align:left; border-radius:8px; }
</style>

<script>
const CSRF = '<?= csrf_token() ?>';
const BASE = '<?= APP_URL ?>';
let activeLeadId = 0;

const outcomes = [
  { k:'picked_up',      icon:'bi-telephone-fill text-success',  label:'Picked Up',          desc:'Customer answered' },
  { k:'no_answer',      icon:'bi-telephone-x text-secondary',   label:'No Answer',          desc:'No response' },
  { k:'voicemail',      icon:'bi-voicemail text-secondary',      label:'Left Voicemail',     desc:'' },
  { k:'callback',       icon:'bi-arrow-clockwise text-warning',  label:'Callback Requested', desc:'Customer asked to call back' },
  { k:'wrong_number',   icon:'bi-telephone-minus text-danger',   label:'Wrong Number',       desc:'' },
  { k:'do_not_call',    icon:'bi-ban text-danger',               label:'Do Not Call',        desc:'Remove from queue' },
];

function openCallModal(lid, name) {
  activeLeadId = lid;
  document.getElementById('callModalName').textContent = name;
  document.getElementById('callNotes').value = '';
  const wrap = document.getElementById('outcomeButtons');
  wrap.innerHTML = '';
  outcomes.forEach(o => {
    const b = document.createElement('button');
    b.className = 'btn btn-outline-secondary outcome-btn py-2 px-3';
    b.innerHTML = `<i class="bi ${o.icon} me-2"></i><span class="fw-medium">${o.label}</span>`
      + (o.desc ? `<span class="text-muted ms-2 small">${o.desc}</span>` : '');
    b.onclick = () => logCall(lid, o.k);
    wrap.appendChild(b);
  });
  new bootstrap.Modal(document.getElementById('callModal')).show();
}

async function logCall(lid, outcome) {
  const notes = document.getElementById('callNotes').value;
  try {
    const r = await fetch(window.location.pathname, {
      method:'POST',
      headers:{'Content-Type':'application/x-www-form-urlencoded'},
      body: 'log_call=1&lead_id='+lid+'&outcome='+outcome+'&notes='+encodeURIComponent(notes)+'&csrf_token='+encodeURIComponent(CSRF)
    });
    const d = await r.json();
    if (!d.ok) { alert('Error: '+(d.error||'Unknown')); return; }

    bootstrap.Modal.getInstance(document.getElementById('callModal'))?.hide();

    // Visual feedback on card
    const card = document.getElementById('card-'+lid);
    if (card) {
      // Show outcome flash
      const flash = document.createElement('div');
      flash.className = 'alert alert-success py-1 px-2 small text-center mt-2';
      flash.textContent = 'Logged: '+outcome.replace(/_/g,' ');
      card.querySelector('.card-body').appendChild(flash);

      if (['do_not_call','wrong_number'].includes(outcome)) {
        setTimeout(()=>card.closest('.queue-card-wrap')?.remove(), 1200);
      }
      if (outcome==='picked_up') {
        // Highlight and navigate to edit
        card.querySelector('.queue-card').style.boxShadow='0 0 0 3px var(--bs-success)';
        setTimeout(()=>{ window.location.href=BASE+'/modules/leads/edit.php?id='+lid+'&from=queue'; },800);
      }
    }
  } catch(e) { alert('Network error: '+e.message); }
}

let forwardLeadId = 0;
function forwardLead(lid, name) {
  forwardLeadId = lid;
  document.getElementById('forwardName').textContent = name;
  new bootstrap.Modal(document.getElementById('forwardModal')).show();
}
document.getElementById('forwardConfirmBtn').addEventListener('click', async () => {
  try {
    const r = await fetch(BASE+'/modules/leads/view.php', {
      method:'POST', headers:{'Content-Type':'application/x-www-form-urlencoded'},
      body:'action=forward&id='+forwardLeadId+'&csrf_token='+encodeURIComponent(CSRF)+'&ajax=1'
    });
    bootstrap.Modal.getInstance(document.getElementById('forwardModal'))?.hide();
    const card = document.getElementById('card-'+forwardLeadId);
    if (card) {
      card.querySelector('.queue-card').classList.add('dimmed');
      const footer = card.querySelector('.card-footer');
      if (footer) footer.innerHTML = '<span class="text-info small w-100 text-center"><i class="bi bi-check2-all me-1"></i>Forwarded to Processing</span>';
    }
    setTimeout(()=>card?.closest('.queue-card-wrap')?.remove(), 1500);
  } catch(e) { alert('Error: '+e.message); }
});
</script>

<?php include __DIR__.'/../../includes/layout_footer.php'; ?>
