<?php
require_once __DIR__.'/../../config/config.php';
require_once __DIR__.'/../../includes/db.php';
require_once __DIR__.'/../../includes/auth.php';
require_once __DIR__.'/../../includes/helpers.php';
session_start_secure();
require_login();

$user = current_user();
$db   = db();

// ── Filters ────────────────────────────────────────────────
$search   = trim($_GET['q']        ?? '');
$status   = trim($_GET['status']   ?? '');
$agentF   = trim($_GET['agent']    ?? '');
$bin      = trim($_GET['bin']      ?? '');
$dateFrom = trim($_GET['date_from']?? '');
$dateTo   = trim($_GET['date_to']  ?? '');
$sort     = safe_sort_col($_GET['sort'] ?? '');
$dir      = ($_GET['dir'] ?? 'desc') === 'asc' ? 'ASC' : 'DESC';
$page     = max(1, (int)($_GET['page'] ?? 1));
$perPage  = max(10, min(200, (int)($_GET['per_page'] ?? 50)));

$customFields     = get_custom_fields();
$searchableFields = array_filter($customFields, fn($f) => $f['searchable']);

// ── Build WHERE + JOINs ────────────────────────────────────
$joins  = '';
$where  = ['1'];
$params = [];

if ($user['role'] === 'agent') { $where[] = 'l.agent_id = ?'; $params[] = $user['id']; }
if ($search) {
    $s = '%'.$search.'%';
    $where[] = '(l.first_name LIKE ? OR l.last_name LIKE ? OR l.email LIKE ? OR l.phone LIKE ? OR l.ref_number LIKE ? OR l.card_issuer LIKE ?)';
    array_push($params, $s, $s, $s, $s, $s, $s);
}
if ($status)   { $where[] = 'l.status = ?';             $params[] = $status;  }
if ($agentF)   { $where[] = 'l.agent_id = ?';           $params[] = $agentF;  }
if ($bin)      { $where[] = 'l.card_number_bin LIKE ?';  $params[] = '%'.$bin.'%'; }
if ($dateFrom) { $where[] = 'DATE(l.created_at) >= ?';   $params[] = $dateFrom; }
if ($dateTo)   { $where[] = 'DATE(l.created_at) <= ?';   $params[] = $dateTo;   }

$cfJoinIdx = 0;
foreach ($searchableFields as $cf) {
    $cfVal = trim($_GET['cf_'.$cf['field_key']] ?? '');
    if ($cfVal === '') continue;
    $alias = 'lcv'.$cfJoinIdx++;
    $joins .= " LEFT JOIN lead_custom_values $alias ON $alias.lead_id=l.id AND $alias.custom_field_id={$cf['id']}";
    $where[] = "$alias.value LIKE ?";
    $params[] = '%'.$cfVal.'%';
}
$whereStr = implode(' AND ', $where);

// ── CSV Export ─────────────────────────────────────────────
if (!empty($_GET['export']) && $_GET['export'] === 'csv') {
    $exportSt = $db->prepare("SELECT l.*, u.full_name AS agent_name, u.username AS agent_username
        FROM leads l LEFT JOIN users u ON u.id=l.agent_id $joins WHERE $whereStr ORDER BY l.created_at DESC LIMIT 100000");
    $exportSt->execute($params);
    $exportLeads = $exportSt->fetchAll();
    if ($exportLeads && $customFields) {
        $ids = implode(',', array_column($exportLeads,'id'));
        $cvRows = $db->query("SELECT lcv.lead_id,cf.field_key,lcv.value FROM lead_custom_values lcv
            JOIN custom_fields cf ON cf.id=lcv.custom_field_id WHERE lcv.lead_id IN ($ids)")->fetchAll();
        $cvMap = [];
        foreach ($cvRows as $r) $cvMap[$r['lead_id']][$r['field_key']] = $r['value'];
        foreach ($exportLeads as &$l) foreach ($customFields as $cf) $l['cv_'.$cf['field_key']] = $cvMap[$l['id']][$cf['field_key']] ?? '';
        unset($l);
    }
    audit('leads_export','lead',null,'Exported '.count($exportLeads).' leads');
    export_leads_csv($exportLeads, $customFields);
}

// ── Count + paginate ───────────────────────────────────────
$cSt = $db->prepare("SELECT COUNT(*) FROM leads l $joins WHERE $whereStr");
$cSt->execute($params);
$total  = (int)$cSt->fetchColumn();
$pages  = max(1, (int)ceil($total/$perPage));
$page   = min($page,$pages);
$offset = ($page-1)*$perPage;

$st = $db->prepare("SELECT l.*, u.full_name AS agent_name, u.username AS agent_username
    FROM leads l LEFT JOIN users u ON u.id=l.agent_id $joins WHERE $whereStr
    ORDER BY l.$sort $dir LIMIT $perPage OFFSET $offset");
$st->execute($params);
$leads = $st->fetchAll();

// Attach custom list-column values
$listCFs = array_filter($customFields, fn($f) => $f['show_in_list']);
if ($leads && $listCFs) {
    $ids = implode(',', array_column($leads,'id'));
    $cvRows = $db->query("SELECT lcv.lead_id,cf.field_key,lcv.value FROM lead_custom_values lcv
        JOIN custom_fields cf ON cf.id=lcv.custom_field_id WHERE lcv.lead_id IN ($ids)")->fetchAll();
    $cvMap = [];
    foreach ($cvRows as $r) $cvMap[$r['lead_id']][$r['field_key']] = $r['value'];
    foreach ($leads as &$l) foreach ($listCFs as $cf) $l['cv_'.$cf['field_key']] = $cvMap[$l['id']][$cf['field_key']] ?? '';
    unset($l);
}

$agents     = $db->query("SELECT id,full_name,username FROM users WHERE role IN ('agent','admin') AND is_active=1 ORDER BY full_name")->fetchAll();
$processors = $db->query("SELECT id,full_name,username FROM users WHERE role IN ('processor','admin') AND is_active=1 ORDER BY full_name")->fetchAll();

$hasFilters = $search||$status||$agentF||$bin||$dateFrom||$dateTo||
    count(array_filter(array_map(fn($f)=>trim($_GET['cf_'.$f['field_key']]??''),$searchableFields)));

$pageTitle = 'Leads';
include __DIR__.'/../../includes/layout_header.php';
?>

<div class="page-header d-flex align-items-center justify-content-between flex-wrap gap-2">
  <div>
    <h1><i class="bi bi-people me-2 text-primary fs-4"></i>Leads</h1>
    <p class="text-muted small mb-0"><?= number_format($total) ?> lead<?= $total!=1?'s':'' ?> found</p>
  </div>
  <div class="d-flex gap-2 flex-wrap">
    <?php if(can_bulk_upload()): ?>
    <a href="<?= APP_URL ?>/modules/leads/bulk_upload.php" class="btn btn-outline-secondary btn-sm">
      <i class="bi bi-cloud-upload me-1"></i>Bulk Upload</a>
    <?php endif; ?>
    <a href="?<?= http_build_query(array_merge($_GET,['export'=>'csv'])) ?>" class="btn btn-outline-success btn-sm">
      <i class="bi bi-download me-1"></i>Export CSV</a>
    <a href="<?= APP_URL ?>/modules/leads/add.php" class="btn btn-primary btn-sm">
      <i class="bi bi-person-plus me-1"></i>Add Lead</a>
  </div>
</div>

<!-- Filter bar -->
<div class="filter-bar mb-3">
  <form method="get" id="filterForm">
    <div class="row g-2 align-items-end">
      <div class="col-12 col-md-3">
        <label class="form-label mb-1 small">Search</label>
        <div class="input-group input-group-sm">
          <span class="input-group-text"><i class="bi bi-search"></i></span>
          <input type="text" name="q" class="form-control" placeholder="Name, email, phone, ref, bank…" value="<?= h($search) ?>">
        </div>
      </div>
      <div class="col-6 col-md-2">
        <label class="form-label mb-1 small">Status</label>
        <select name="status" class="form-select form-select-sm">
          <option value="">All Statuses</option>
          <?php foreach(['new','in_review','approved','declined','forwarded','on_hold'] as $s): ?>
            <option value="<?= $s ?>" <?= $status===$s?'selected':'' ?>><?= ucfirst(str_replace('_',' ',$s)) ?></option>
          <?php endforeach; ?>
        </select>
      </div>
      <?php if($user['role']!=='agent'): ?>
      <div class="col-6 col-md-2">
        <label class="form-label mb-1 small">Agent</label>
        <select name="agent" class="form-select form-select-sm">
          <option value="">All Agents</option>
          <?php foreach($agents as $a): ?>
            <option value="<?= $a['id'] ?>" <?= $agentF==$a['id']?'selected':'' ?>><?= h($a['full_name']?:$a['username']) ?></option>
          <?php endforeach; ?>
        </select>
      </div>
      <?php endif; ?>
      <div class="col-6 col-md-2">
        <label class="form-label mb-1 small">BIN</label>
        <input type="text" name="bin" class="form-control form-control-sm" placeholder="BIN…" value="<?= h($bin) ?>">
      </div>
      <div class="col-6 col-md-2">
        <label class="form-label mb-1 small">Added From</label>
        <input type="date" name="date_from" class="form-control form-control-sm" value="<?= h($dateFrom) ?>">
      </div>
      <div class="col-6 col-md-2">
        <label class="form-label mb-1 small">Added To</label>
        <input type="date" name="date_to" class="form-control form-control-sm" value="<?= h($dateTo) ?>">
      </div>
      <!-- Searchable custom fields -->
      <?php foreach($searchableFields as $cf):
        $cfVal=trim($_GET['cf_'.$cf['field_key']]??'');
      ?>
      <div class="col-6 col-md-2">
        <label class="form-label mb-1 small"><?= h($cf['label']) ?></label>
        <?php if($cf['field_type']==='select'): ?>
          <select name="cf_<?= h($cf['field_key']) ?>" class="form-select form-select-sm">
            <option value="">Any</option>
            <?php foreach(json_decode($cf['options']??'[]',true)?:[] as $opt): ?>
              <option value="<?= h($opt) ?>" <?= $cfVal===$opt?'selected':'' ?>><?= h($opt) ?></option>
            <?php endforeach; ?>
          </select>
        <?php else: ?>
          <input type="<?= $cf['field_type']==='date'?'date':'text' ?>"
                 name="cf_<?= h($cf['field_key']) ?>" class="form-control form-control-sm"
                 placeholder="Filter…" value="<?= h($cfVal) ?>">
        <?php endif; ?>
      </div>
      <?php endforeach; ?>
      <div class="col-auto d-flex gap-1 mt-auto">
        <button type="submit" class="btn btn-primary btn-sm"><i class="bi bi-funnel me-1"></i>Filter</button>
        <?php if($hasFilters): ?>
          <a href="<?= APP_URL ?>/modules/leads/index.php" class="btn btn-outline-secondary btn-sm"><i class="bi bi-x"></i></a>
        <?php endif; ?>
      </div>
      <div class="col-auto ms-auto mt-auto">
        <select name="per_page" class="form-select form-select-sm" onchange="this.form.submit()" title="Rows per page">
          <?php foreach([25,50,100,200] as $pp): ?>
            <option value="<?= $pp ?>" <?= $perPage==$pp?'selected':'' ?>><?= $pp ?>/page</option>
          <?php endforeach; ?>
        </select>
      </div>
    </div>
  </form>
</div>

<!-- Batch action bar -->
<div id="batchBar" class="alert alert-secondary py-2 mb-3" style="display:none">
  <div class="d-flex align-items-center gap-3 flex-wrap">
    <span class="small fw-semibold"><span id="batchCount">0</span> selected</span>
    <form method="post" action="<?= APP_URL ?>/modules/leads/batch_action.php" id="batchForm" class="d-flex gap-2 flex-wrap align-items-center">
      <input type="hidden" name="csrf_token" value="<?= csrf_token() ?>">
      <input type="hidden" name="ids" id="batchIds">
      <input type="hidden" name="redirect_url" value="<?= h($_SERVER['REQUEST_URI']) ?>">
      <select name="action" class="form-select form-select-sm" style="width:auto" id="batchAction">
        <option value="">— Batch Action —</option>
        <optgroup label="Status">
          <option value="forward">Forward to Processing</option>
          <option value="status_new">Mark New</option>
          <option value="status_in_review">Mark In Review</option>
          <option value="status_approved">Mark Approved</option>
          <option value="status_declined">Mark Declined</option>
          <option value="status_on_hold">Mark On Hold</option>
        </optgroup>
        <optgroup label="Assignment">
          <option value="assign">Assign to Processor…</option>
        </optgroup>
        <?php if($customFields): ?>
        <optgroup label="Edit Field">
          <?php foreach($customFields as $cf): ?>
          <option value="bulk_field_<?= h($cf['field_key']) ?>">Set: <?= h($cf['label']) ?>…</option>
          <?php endforeach; ?>
        </optgroup>
        <?php endif; ?>
        <optgroup label="Export">
          <option value="export_selected">Export Selected as CSV</option>
        </optgroup>
        <?php if(in_array($user['role'],['admin','processor'])): ?>
        <optgroup label="Danger">
          <option value="delete">Delete Selected</option>
        </optgroup>
        <?php endif; ?>
      </select>
      <button type="submit" class="btn btn-sm btn-secondary">Apply</button>
    </form>
  </div>
</div>

<!-- Leads table -->
<div class="card p-0">
  <div class="leads-scroll-wrap">
    <table class="table table-hover mb-0">
      <thead class="sticky-top">
        <tr>
          <th style="width:36px"><input type="checkbox" class="form-check-input" id="selectAll"></th>
          <?php
          function sort_link(string $col, string $label, string $cur, string $curDir, array $get): string {
            $active = $cur===$col;
            $nextDir = ($active && $curDir==='ASC') ? 'desc' : 'asc';
            $href = '?'.http_build_query(array_merge($get,['sort'=>$col,'dir'=>$nextDir,'page'=>1]));
            $arrow = $active ? ($curDir==='ASC'?'↑':'↓') : '<span class="opacity-25">↕</span>';
            return "<a href=\"$href\" class=\"text-decoration-none ".($active?'text-primary':'')."\">{$label} {$arrow}</a>";
          }
          ?>
          <th><?= sort_link('last_name','Name',$sort,$dir,$_GET) ?></th>
          <th>Phone</th><th>Email</th><th>Bank / BIN</th>
          <th><?= sort_link('status','Status',$sort,$dir,$_GET) ?></th>
          <?php if($user['role']!=='agent'): ?><th>Agent</th><?php endif; ?>
          <th><?= sort_link('created_at','Added',$sort,$dir,$_GET) ?></th>
          <?php foreach($listCFs as $cf): ?>
            <th class="small"><?= h($cf['label']) ?></th>
          <?php endforeach; ?>
          <th style="width:80px">Actions</th>
        </tr>
      </thead>
      <tbody>
        <?php if($leads): foreach($leads as $l): ?>
        <tr>
          <td><input type="checkbox" class="form-check-input row-check" value="<?= $l['id'] ?>"></td>
          <td>
            <a href="<?= APP_URL ?>/modules/leads/view.php?id=<?= $l['id'] ?>" class="fw-medium text-decoration-none">
              <?= h($l['first_name'].' '.$l['last_name']) ?></a>
            <div class="mono text-muted" style="font-size:.68rem"><?= h($l['ref_number']) ?></div>
          </td>
          <td class="mono small"><?= h($l['phone']?format_phone($l['phone']):'—') ?></td>
          <td style="max-width:160px" class="text-truncate small"><?= h($l['email']?:'—') ?></td>
          <td>
            <?php if($l['card_issuer']): ?><div class="small fw-medium"><?= h($l['card_issuer']) ?></div><?php endif; ?>
            <?php if($l['card_number_bin']): ?><div class="mono text-muted" style="font-size:.68rem"><?= h($l['card_number_bin']) ?></div><?php endif; ?>
            <?php if(!$l['card_issuer']&&!$l['card_number_bin']): ?>—<?php endif; ?>
          </td>
          <td><?= status_badge($l['status']) ?></td>
          <?php if($user['role']!=='agent'): ?>
          <td class="small text-muted"><?= h($l['agent_name']?:$l['agent_username']?:'—') ?></td>
          <?php endif; ?>
          <td class="small text-muted"><?= h(date('M j, Y',strtotime($l['created_at']))) ?></td>
          <?php foreach($listCFs as $cf): ?>
            <td class="small"><?= h($l['cv_'.$cf['field_key']]??'—') ?></td>
          <?php endforeach; ?>
          <td>
            <a href="<?= APP_URL ?>/modules/leads/view.php?id=<?= $l['id'] ?>" class="btn btn-sm btn-outline-secondary p-1 lh-1" title="View"><i class="bi bi-eye"></i></a>
            <a href="<?= APP_URL ?>/modules/leads/edit.php?id=<?= $l['id'] ?>" class="btn btn-sm btn-outline-primary p-1 lh-1" title="Edit"><i class="bi bi-pencil"></i></a>
          </td>
        </tr>
        <?php endforeach; else: ?>
        <tr><td colspan="<?= 8+($user['role']!=='agent'?1:0)+count($listCFs) ?>" class="text-center text-muted py-5">
          <i class="bi bi-search fs-3 d-block mb-2 opacity-25"></i>
          No leads found<?= $hasFilters?' matching your filters':'' ?>.
          <?php if($hasFilters): ?>
          <div class="mt-2"><a href="<?= APP_URL ?>/modules/leads/index.php" class="btn btn-sm btn-outline-secondary">Clear filters</a></div>
          <?php endif; ?>
        </td></tr>
        <?php endif; ?>
      </tbody>
    </table>
  </div>

  <!-- Pagination -->
  <div class="card-footer d-flex align-items-center justify-content-between py-2 flex-wrap gap-2">
    <small class="text-muted">
      <?php if($total>0): ?>Showing <?= number_format(($page-1)*$perPage+1) ?>–<?= number_format(min($page*$perPage,$total)) ?> of <?= number_format($total) ?>
      <?php else: ?>0 results<?php endif; ?>
    </small>
    <?php if($pages>1): ?>
    <nav><ul class="pagination pagination-sm mb-0 gap-1">
      <?php $qp=$_GET; unset($qp['page']); $base='?'.http_build_query($qp).'&page=';
      if($page>1): ?><li class="page-item"><a class="page-link" href="<?= $base.($page-1) ?>">‹</a></li><?php endif; ?>
      <?php $s2=max(1,$page-2);$e2=min($pages,$page+2);
      if($s2>1){echo '<li class="page-item"><a class="page-link" href="'.$base.'1">1</a></li>';
        if($s2>2) echo '<li class="page-item disabled"><span class="page-link">…</span></li>';}
      for($i=$s2;$i<=$e2;$i++): ?>
        <li class="page-item <?= $i===$page?'active':'' ?>"><a class="page-link" href="<?= $base.$i ?>"><?= $i ?></a></li>
      <?php endfor;
      if($e2<$pages){if($e2<$pages-1) echo '<li class="page-item disabled"><span class="page-link">…</span></li>';
        echo '<li class="page-item"><a class="page-link" href="'.$base.$pages.'">'.$pages.'</a></li>';}
      if($page<$pages): ?><li class="page-item"><a class="page-link" href="<?= $base.($page+1) ?>">›</a></li><?php endif; ?>
    </ul></nav>
    <?php endif; ?>
  </div>
</div>

<!-- FAB -->
<a href="<?= APP_URL ?>/modules/leads/add.php" class="btn btn-primary btn-fab" title="Add New Lead"><i class="bi bi-plus-lg"></i></a>

<!-- Assign Modal -->
<div class="modal fade" id="assignModal" tabindex="-1">
  <div class="modal-dialog modal-sm"><div class="modal-content">
    <div class="modal-header"><h6 class="modal-title mb-0">Assign to Processor</h6>
      <button type="button" class="btn-close" data-bs-dismiss="modal"></button></div>
    <div class="modal-body">
      <select id="assignSelect" class="form-select">
        <option value="">— Unassigned —</option>
        <?php foreach($processors as $p): ?>
          <option value="<?= $p['id'] ?>"><?= h($p['full_name']?:$p['username']) ?></option>
        <?php endforeach; ?>
      </select>
    </div>
    <div class="modal-footer">
      <button class="btn btn-primary btn-sm" id="assignConfirm">Assign</button>
    </div>
  </div></div>
</div>

<!-- Bulk field edit modal -->
<div class="modal fade" id="bulkFieldModal" tabindex="-1">
  <div class="modal-dialog modal-sm"><div class="modal-content">
    <div class="modal-header"><h6 class="modal-title mb-0" id="bulkFieldTitle">Set Field Value</h6>
      <button type="button" class="btn-close" data-bs-dismiss="modal"></button></div>
    <div class="modal-body"><div id="bulkFieldInput"></div></div>
    <div class="modal-footer">
      <button class="btn btn-primary btn-sm" id="bulkFieldConfirm">Apply to Selected</button>
    </div>
  </div></div>
</div>

<script>
const batchBar   = document.getElementById('batchBar');
const batchCount = document.getElementById('batchCount');
const batchIds   = document.getElementById('batchIds');
const batchAction= document.getElementById('batchAction');
const batchForm  = document.getElementById('batchForm');

function updateBatch() {
  const checked = [...document.querySelectorAll('.row-check:checked')];
  batchBar.style.display = checked.length ? 'block' : 'none';
  if (checked.length) {
    batchCount.textContent = checked.length;
    batchIds.value = checked.map(c=>c.value).join(',');
  }
}
document.querySelectorAll('.row-check').forEach(cb=>cb.addEventListener('change',updateBatch));
document.getElementById('selectAll').addEventListener('change',function(){
  document.querySelectorAll('.row-check').forEach(cb=>cb.checked=this.checked); updateBatch();
});

batchForm?.addEventListener('submit',function(e){
  const action=batchAction.value;
  if(!action){e.preventDefault();alert('Please select a batch action.');return;}
  if(action==='assign'){e.preventDefault();new bootstrap.Modal(document.getElementById('assignModal')).show();return;}
  if(action.startsWith('bulk_field_')){e.preventDefault();openBulkFieldModal(action.replace('bulk_field_',''));return;}
  if(action==='export_selected'){
    e.preventDefault();
    window.location.href='<?= APP_URL ?>/modules/leads/export_selected.php?ids='+encodeURIComponent(batchIds.value)+'&csrf=<?= csrf_token() ?>';
    return;
  }
  if(action==='delete'&&!confirm('Delete selected leads? This cannot be undone.')){e.preventDefault();}
});

document.getElementById('assignConfirm')?.addEventListener('click',()=>{
  const val=document.getElementById('assignSelect').value;
  addHidden('assign_to',val); batchAction.value='assign';
  bootstrap.Modal.getInstance(document.getElementById('assignModal')).hide();
  batchForm.submit();
});

const cfMeta=<?= json_encode(array_values($customFields)) ?>;

function openBulkFieldModal(key){
  const cf=cfMeta.find(f=>f.field_key===key); if(!cf)return;
  document.getElementById('bulkFieldTitle').textContent='Set: '+cf.label;
  const wrap=document.getElementById('bulkFieldInput'); wrap.innerHTML='';
  let inp;
  if(cf.field_type==='select'){
    inp=document.createElement('select'); inp.className='form-select';
    const bl=document.createElement('option'); bl.value=''; bl.textContent='— Select —'; inp.appendChild(bl);
    (JSON.parse(cf.options||'[]')).forEach(o=>{const op=document.createElement('option');op.value=o;op.textContent=o;inp.appendChild(op);});
  } else if(cf.field_type==='textarea'){
    inp=document.createElement('textarea'); inp.className='form-control'; inp.rows=3;
  } else {
    inp=document.createElement('input');
    inp.type=cf.field_type==='number'?'number':cf.field_type==='date'?'date':'text';
    inp.className='form-control';
  }
  inp.id='bulkFieldValue'; wrap.appendChild(inp);
  new bootstrap.Modal(document.getElementById('bulkFieldModal')).show();
  document.getElementById('bulkFieldConfirm').onclick=()=>{
    const val=document.getElementById('bulkFieldValue').value;
    addHidden('bulk_field_key',key); addHidden('bulk_field_value',val);
    batchAction.value='bulk_field_'+key;
    bootstrap.Modal.getInstance(document.getElementById('bulkFieldModal')).hide();
    batchForm.submit();
  };
}

function addHidden(name,val){
  const h=document.createElement('input'); h.type='hidden'; h.name=name; h.value=val;
  batchForm.appendChild(h);
}
</script>

<?php include __DIR__.'/../../includes/layout_footer.php'; ?>
