<?php
require_once __DIR__.'/../../config/config.php';
require_once __DIR__.'/../../includes/db.php';
require_once __DIR__.'/../../includes/auth.php';
require_once __DIR__.'/../../includes/helpers.php';
session_start_secure();
require_role('admin');

$db = db();
$errors = [];
$editUser = null;

// Handle form submissions
if ($_SERVER['REQUEST_METHOD']==='POST') {
    csrf_check();
    $action = $_POST['action']??'';

    if ($action==='save') {
        $uid       = (int)($_POST['user_id']??0);
        $username  = trim($_POST['username']??'');
        $email     = trim($_POST['email']??'');
        $role      = $_POST['role']??'agent';
        $fullName  = trim($_POST['full_name']??'');
        $password  = $_POST['password']??'';
        $isActive  = isset($_POST['is_active'])?1:0;

        if (!$username) $errors[]='Username required.';
        if (!$email || !filter_var($email,FILTER_VALIDATE_EMAIL)) $errors[]='Valid email required.';
        if (!$uid && !$password) $errors[]='Password required for new users.';
        if ($password && strlen($password)<8) $errors[]='Password must be at least 8 characters.';

        if (!$errors) {
            if ($uid) {
                $sql = "UPDATE users SET username=?,email=?,role=?,full_name=?,is_active=?";
                $params = [$username,$email,$role,$fullName,$isActive];
                if ($password) { $sql.=',password_hash=?'; $params[]=password_hash($password,PASSWORD_BCRYPT,['cost'=>12]); }
                $sql.=' WHERE id=?'; $params[]=$uid;
                $db->prepare($sql)->execute($params);
                audit('user_updated','user',$uid);
                flash('success','User updated.');
            } else {
                $hash = password_hash($password,PASSWORD_BCRYPT,['cost'=>12]);
                $st = $db->prepare("INSERT INTO users (username,email,password_hash,role,full_name,is_active) VALUES (?,?,?,?,?,?)");
                $st->execute([$username,$email,$hash,$role,$fullName,$isActive]);
                audit('user_created','user',(int)$db->lastInsertId());
                flash('success','User created.');
            }
            redirect(APP_URL.'/modules/users/index.php');
        }
    }

    if ($action==='deactivate') {
        $uid=(int)$_POST['user_id'];
        $db->prepare("UPDATE users SET is_active=0 WHERE id=? AND id!=?")->execute([$uid,current_user()['id']]);
        audit('user_deactivated','user',$uid);
        flash('success','User deactivated.');
        redirect(APP_URL.'/modules/users/index.php');
    }
    if ($action==='activate') {
        $uid=(int)$_POST['user_id'];
        $db->prepare("UPDATE users SET is_active=1 WHERE id=?")->execute([$uid]);
        audit('user_activated','user',$uid);
        flash('success','User activated.');
        redirect(APP_URL.'/modules/users/index.php');
    }
    if ($action==='force_logout') {
        // Just audit — actual session invalidation would need DB sessions
        $uid=(int)$_POST['user_id'];
        audit('user_force_logout','user',$uid,'Forced by admin');
        flash('info','Force logout logged. If using DB sessions, the user session will expire.');
        redirect(APP_URL.'/modules/users/index.php');
    }
}

if (isset($_GET['edit'])) {
    $st=$db->prepare("SELECT * FROM users WHERE id=?");
    $st->execute([(int)$_GET['edit']]);
    $editUser=$st->fetch();
}

$users = $db->query("SELECT * FROM users ORDER BY role,full_name,username")->fetchAll();
$pageTitle = 'User Management';
include __DIR__.'/../../includes/layout_header.php';
?>

<div class="page-header d-flex align-items-center justify-content-between">
  <h1><i class="bi bi-person-gear me-2 text-primary fs-4"></i>User Management</h1>
  <button class="btn btn-primary btn-sm" data-bs-toggle="modal" data-bs-target="#userModal">
    <i class="bi bi-person-plus me-1"></i>Add User</button>
</div>

<?php if($errors): ?>
<div class="alert alert-danger"><?= implode('<br>',$errors) ?></div>
<?php endif; ?>

<div class="card">
  <div class="table-responsive">
    <table class="table table-hover mb-0">
      <thead>
        <tr>
          <th>Name</th>
          <th>Username</th>
          <th>Email</th>
          <th>Role</th>
          <th>Status</th>
          <th>Last Login</th>
          <th>Actions</th>
        </tr>
      </thead>
      <tbody>
        <?php foreach($users as $u): ?>
        <tr>
          <td><?= h($u['full_name']?:'—') ?></td>
          <td class="mono"><?= h($u['username']) ?></td>
          <td><?= h($u['email']) ?></td>
          <td><span class="badge bg-secondary text-capitalize"><?= h($u['role']) ?></span></td>
          <td><?= $u['is_active']
            ? '<span class="status-badge badge-approved">Active</span>'
            : '<span class="status-badge badge-declined">Inactive</span>' ?></td>
          <td class="text-muted small"><?= $u['last_login'] ? h(date('M j, Y g:ia',strtotime($u['last_login']))) : 'Never' ?></td>
          <td>
            <button class="btn btn-sm btn-outline-primary p-1 lh-1" onclick="openEditModal(<?= htmlspecialchars(json_encode($u)) ?>)"
              title="Edit"><i class="bi bi-pencil"></i></button>
            <form method="post" class="d-inline">
              <input type="hidden" name="csrf_token" value="<?= csrf_token() ?>">
              <input type="hidden" name="user_id" value="<?= $u['id'] ?>">
              <input type="hidden" name="action" value="<?= $u['is_active']?'deactivate':'activate' ?>">
              <?php if($u['id']!=current_user()['id']): ?>
              <button type="submit" class="btn btn-sm btn-outline-<?= $u['is_active']?'warning':'success' ?> p-1 lh-1"
                title="<?= $u['is_active']?'Deactivate':'Activate' ?>">
                <i class="bi bi-<?= $u['is_active']?'person-x':'person-check' ?>"></i></button>
              <?php endif; ?>
            </form>
            <form method="post" class="d-inline">
              <input type="hidden" name="csrf_token" value="<?= csrf_token() ?>">
              <input type="hidden" name="user_id" value="<?= $u['id'] ?>">
              <input type="hidden" name="action" value="force_logout">
              <button type="submit" class="btn btn-sm btn-outline-secondary p-1 lh-1" title="Force Logout">
                <i class="bi bi-box-arrow-right"></i></button>
            </form>
          </td>
        </tr>
        <?php endforeach; ?>
      </tbody>
    </table>
  </div>
</div>

<!-- User Modal -->
<div class="modal fade" id="userModal" tabindex="-1">
  <div class="modal-dialog">
    <div class="modal-content">
      <form method="post" id="userForm">
        <input type="hidden" name="csrf_token" value="<?= csrf_token() ?>">
        <input type="hidden" name="action" value="save">
        <input type="hidden" name="user_id" id="modal_user_id" value="0">
        <div class="modal-header">
          <h5 class="modal-title" id="userModalTitle">Add User</h5>
          <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
        </div>
        <div class="modal-body">
          <div class="row g-3">
            <div class="col-md-6">
              <label class="form-label">Full Name</label>
              <input type="text" name="full_name" id="modal_full_name" class="form-control">
            </div>
            <div class="col-md-6">
              <label class="form-label">Username *</label>
              <input type="text" name="username" id="modal_username" class="form-control" required>
            </div>
            <div class="col-12">
              <label class="form-label">Email *</label>
              <input type="email" name="email" id="modal_email" class="form-control" required>
            </div>
            <div class="col-md-6">
              <label class="form-label">Password <span id="pwHint" class="text-muted small">(required)</span></label>
              <input type="password" name="password" id="modal_password" class="form-control" minlength="8">
            </div>
            <div class="col-md-6">
              <label class="form-label">Role *</label>
              <select name="role" id="modal_role" class="form-select">
                <option value="agent">Agent</option>
                <option value="processor">Processor</option>
                <option value="admin">Admin</option>
              </select>
            </div>
            <div class="col-12">
              <div class="form-check">
                <input type="checkbox" name="is_active" id="modal_is_active" class="form-check-input" value="1" checked>
                <label class="form-check-label" for="modal_is_active">Active</label>
              </div>
            </div>
          </div>
        </div>
        <div class="modal-footer">
          <button type="button" class="btn btn-outline-secondary" data-bs-dismiss="modal">Cancel</button>
          <button type="submit" class="btn btn-primary">Save User</button>
        </div>
      </form>
    </div>
  </div>
</div>

<script>
function openEditModal(u) {
  document.getElementById('userModalTitle').textContent = 'Edit User: ' + u.username;
  document.getElementById('modal_user_id').value  = u.id;
  document.getElementById('modal_full_name').value = u.full_name || '';
  document.getElementById('modal_username').value  = u.username;
  document.getElementById('modal_email').value     = u.email;
  document.getElementById('modal_role').value      = u.role;
  document.getElementById('modal_password').value  = '';
  document.getElementById('modal_password').required = false;
  document.getElementById('modal_is_active').checked = u.is_active == 1;
  document.getElementById('pwHint').textContent = '(leave blank to keep)';
  new bootstrap.Modal(document.getElementById('userModal')).show();
}
// Reset for new user
document.querySelector('[data-bs-target="#userModal"]')?.addEventListener('click',()=>{
  document.getElementById('userModalTitle').textContent='Add User';
  document.getElementById('userForm').reset();
  document.getElementById('modal_user_id').value='0';
  document.getElementById('modal_password').required=true;
  document.getElementById('pwHint').textContent='(required)';
  document.getElementById('modal_is_active').checked=true;
});
</script>

<?php include __DIR__.'/../../includes/layout_footer.php'; ?>
